package com.turbospaces.executor;

import java.util.LinkedList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ExecutorService;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.MDC;

import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.BootstrapAware;

import io.netty.util.AsciiString;
import io.vavr.CheckedRunnable;

public class SerialContextWorker extends AbstractBootstrapAware implements ContextWorker {
    private final Object mutex = new Object();
    private final Queue<Runnable> queue = new LinkedList<>();
    private final AsciiString key;
    private final ExecutorService executor;
    private Runnable active;

    public SerialContextWorker(AsciiString key, ExecutorService executor) {
        this.key = Objects.requireNonNull(key);
        this.executor = Objects.requireNonNull(executor);
    }
    @Override
    public ExecutorService executor() {
        return executor;
    }
    @Override
    public ContextWorker forKey(WorkUnit unit) {
        throw new UnsupportedOperationException();
    }
    @Override
    public void schedule(CheckedRunnable command) {
        if (command instanceof AbstractBootstrapAware) {
            ((BootstrapAware) command).setBootstrap(bootstrap);
        }

        synchronized (mutex) {
            long now = System.currentTimeMillis();
            Map<String, String> mdc = MDC.getCopyOfContextMap(); // capture MDC
            queue.add(new Runnable() {
                @Override
                public void run() {
                    Thread currentThread = Thread.currentThread();
                    String oldName = currentThread.getName();
                    String newName = oldName + "|" + key;
                    currentThread.setName(newName);

                    try {
                        if (mdc != null) {
                            for (Entry<String, String> it : mdc.entrySet()) {
                                MDC.put(it.getKey(), it.getValue());
                            }
                        }

                        long delta = System.currentTimeMillis() - now;
                        if (delta > 0) {
                            logger.debug("task({}) was queued for {} ms before exec for key: {}", Math.abs(hashCode()), delta, key);
                        }

                        logger.trace("before apply: {}", command);
                        command.run();
                    } catch (Throwable err) {
                        logger.error(err.getMessage(), err);
                        ExceptionUtils.wrapAndThrow(err);
                    } finally {
                        currentThread.setName(oldName);

                        if (mdc != null) {
                            for (String it : mdc.keySet()) {
                                MDC.remove(it);
                            }
                        }

                        scheduleNext();
                    }
                }
            });
            if (Objects.isNull(active)) {
                scheduleNext();
            }
        }
    }
    private void scheduleNext() {
        synchronized (mutex) {
            if ((active = queue.poll()) != null) {
                logger.debug("submitting task by key: {}, queue size: {}, executor: {}", key, queue.size(), executor);
                executor.execute(active);
            }
        }
    }
}
