package com.turbospaces.boot;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.lang3.time.StopWatch;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.executor.Executors;

import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import io.vavr.CheckedRunnable;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class DefaultPlatformExecutorService extends AbstractBootstrapAware implements PlatformExecutorService {
    private final String name;
    private final int minPoolSize;
    private final int maxPoolSize;

    private ListeningExecutorService executor;

    public DefaultPlatformExecutorService(String name, int minPoolSize, int maxPoolSize) {
        this.name = Objects.requireNonNull(name);
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
    }
    @Override
    public void afterPropertiesSet() {
        ThreadFactoryBuilder factory = new ThreadFactoryBuilder();
        factory.setDaemon(false);
        factory.setNameFormat(name + "-%d");
        ThreadFactory threadFactory = factory.build();

        Duration ttl = bootstrap.props().APP_PLATFORM_MAX_IDLE.get();
        ExecutorService pool = Executors.newScalingThreadPool(minPoolSize, maxPoolSize, ttl, threadFactory);

        ExecutorServiceMetrics metrics = new ExecutorServiceMetrics(pool, name, Collections.emptyList());
        metrics.bindTo(bootstrap.meterRegistry());

        executor = MoreExecutors.listeningDecorator(pool);
    }
    @Override
    public void destroy() {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = bootstrap.props().APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();

        logger.debug("stopped executor-service: {} in {}", executor, stopWatch);
    }
    @Override
    public ListenableFuture<?> execute(CheckedRunnable action) {
        return submit(new Runnable() {
            @Override
            public void run() {
                try {
                    action.run();
                } catch (Throwable err) {
                    log.error(err.getMessage(), err);
                }
            }
        });
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = getMdcContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    public ListenableFuture<?> submit(Runnable task) {
        Map<String, String> mdcContextMap = getMdcContextMap();
        return executor.submit(wrapRunnable(task, mdcContextMap));
    }
    @Override
    public <T> ListenableFuture<T> submit(Callable<T> task) {
        Map<String, String> mdcContextMap = getMdcContextMap();
        return executor.submit(wrapCallable(task, mdcContextMap));
    }
    @Override
    public <T> ListenableFuture<T> submit(Runnable task, T result) {
        Map<String, String> mdcContextMap = getMdcContextMap();
        return executor.submit(wrapRunnable(task, mdcContextMap), result);
    }
    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
        return executor.invokeAll(tasks);
    }
    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException {
        return executor.invokeAll(tasks, timeout, unit);
    }
    @Override
    public void shutdown() {
        executor.shutdown();
    }
    @Override
    public List<Runnable> shutdownNow() {
        return executor.shutdownNow();
    }
    @Override
    public boolean isShutdown() {
        return executor.isShutdown();
    }
    @Override
    public boolean isTerminated() {
        return executor.isTerminated();
    }
    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return executor.awaitTermination(timeout, unit);
    }
    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
        return executor.invokeAny(tasks);
    }
    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        return executor.invokeAny(tasks, timeout, unit);
    }
}
