package com.turbospaces.boot;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;

import javax.inject.Inject;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.AbstractFactoryBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.scheduling.annotation.Scheduled;

import com.netflix.archaius.api.annotations.Configuration;
import com.turbospaces.boot.MockBootstrap.InboundChannel;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.TimedConfiguration;

import io.micrometer.core.annotation.Timed;

@Configuration
@Import({ SchedulingDiModule.class, TimedConfiguration.class })
public class MockConfig extends AbstractBootstrapAware {
    @Bean
    public HttpFactory factory(HttpClientBuilder client) {
        return new HttpFactory(client);
    }
    @Bean
    public Foo foo(HttpFactory factory) throws Exception {
        return new Foo(factory.getObject());
    }
    @Scheduled(fixedRate = 1000)
    public void mock() {
        logger.info("The time is now {}", new SimpleDateFormat("HH:mm:ss").format(new Date()));
    }
    @Bean
    public InboundChannel inboundChannel() throws Exception {
        URIBuilder uri = new URIBuilder().setScheme("http").setHost("localhost").setPath("/v1/status").setPort(bootstrap.port());
        return new InboundChannel(uri.build());
    }

    public static class Foo extends AbstractBootstrapAware implements InitializingBean, DisposableBean {
        @Value(CloudOptions.CLOUD_APP_ID)
        String appId;

        private final CloseableHttpClient client;

        @Inject
        public Foo(CloseableHttpClient client) {
            this.client = Objects.requireNonNull(client);
        }
        @Override
        public void afterPropertiesSet() throws Exception {
            logger.debug("Starting foo bean now ...");
        }
        @Override
        public void destroy() throws Exception {
            logger.debug("Stopping foo bean now ...");
        }
        @Timed
        public void print() throws InterruptedException {
            CountDownLatch l = new CountDownLatch(1);
            bootstrap.globalPlatform().execute(new Runnable() {
                @Override
                public void run() {
                    logger.debug(toString());
                    l.countDown();
                }
            });

            l.await();
        }
        @Override
        public String toString() {
            ToStringBuilder toString = new ToStringBuilder(this, ToStringStyle.NO_CLASS_NAME_STYLE);
            return toString.append("appId", appId).append("client", client).append("version", bootstrap.release()).build();
        }
    }

    public static class HttpFactory extends AbstractFactoryBean<CloseableHttpClient> {
        private final HttpClientBuilder builder;

        public HttpFactory(HttpClientBuilder builder) {
            this.builder = Objects.requireNonNull(builder);
        }
        @Override
        public Class<?> getObjectType() {
            return CloseableHttpClient.class;
        }
        @Override
        protected CloseableHttpClient createInstance() throws Exception {
            return builder.build();
        }
        @Override
        protected void destroyInstance(CloseableHttpClient instance) throws Exception {
            instance.close();
        }
    }
}
