package com.turbospaces.boot;

import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.RawServiceInfo;
import com.turbospaces.ups.UPSs;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.service.BaseServiceInfo;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.common.PostgresqlServiceInfo;
import reactor.core.publisher.Flux;
import reactor.core.scheduler.Scheduler;

import java.time.Duration;
import java.util.Arrays;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;

public class UpssTest {
    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Test
    public void works() throws Exception {
        String ups = UUID.randomUUID().toString();

        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap bootstrap = new SimpleBootstrap(props);
        bootstrap.run();

        try {
            try {
                bootstrap.serviceInfoByName(ups).blockFirst(Duration.ofMillis(1));
                Assertions.fail();
            } catch (RuntimeException err) {

            }

            Scheduler scheduler = bootstrap.globalPlatform().scheduler();
            CountDownLatch latch = new CountDownLatch(1);
            bootstrap.serviceInfoByName(ups).subscribeOn(scheduler).subscribe(new Consumer<ServiceInfo>() {
                @Override
                public void accept(ServiceInfo t) {
                    latch.countDown();
                }
            });

            new Thread(new Runnable() {
                @Override
                public void run() {
                    bootstrap.addUps(new BaseServiceInfo(ups));
                }
            }).start();

            Assertions.assertTrue(latch.await(1, TimeUnit.MINUTES));
            Assertions.assertNotNull(bootstrap.cloud().getServiceInfo(ups));

            bootstrap.removeUps(ups);
            Assertions.assertTrue(bootstrap.cloud().getServiceInfos().isEmpty());

            try {
                bootstrap.serviceInfoByName(ups).timeout(Duration.ofMillis(1)).hasElements().block();
                Assertions.fail();
            } catch (Exception err) {
                Assertions.assertTrue(ExceptionUtils.getRootCause(err).getClass().equals(TimeoutException.class));
            }
        } finally {
            bootstrap.shutdown();
        }
    }

    @Test
    public void scoped() throws Exception {
        String scope = Long.toString(System.currentTimeMillis());
        String ups = UUID.randomUUID().toString();
        UUID uuid = UUID.randomUUID();

        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap bootstrap = new SimpleBootstrap(props);
        bootstrap.run();

        try {
            try {
                bootstrap.scopedServiceInfoByName(scope, ups).blockFirst(Duration.ofMillis(1));
                Assertions.fail();
            } catch (RuntimeException err) {

            }

            Scheduler scheduler = bootstrap.globalPlatform().scheduler();
            CountDownLatch latch = new CountDownLatch(2);
            bootstrap.scopedServiceInfoByName(scope, ups).subscribeOn(scheduler).subscribe(new Consumer<ServiceInfo>() {
                @Override
                public void accept(ServiceInfo t) {
                    latch.countDown();
                }
            });

            new Thread(new Runnable() {
                @Override
                public void run() {
                    bootstrap.addUps(new BaseServiceInfo(ups));
                    bootstrap.addUps(new BaseServiceInfo(uuid + "-" + ups));
                    bootstrap.addUps(new BaseServiceInfo(scope + "-" + ups));
                }
            }).start();

            Assertions.assertTrue(latch.await(1, TimeUnit.MINUTES));
            Assertions.assertNotNull(bootstrap.cloud().getServiceInfo(ups));
            Assertions.assertNotNull(bootstrap.cloud().getServiceInfo(scope + "-" + ups));

            bootstrap.removeUps(ups);
            bootstrap.removeUps(scope + "-" + ups);
            bootstrap.removeUps(uuid + "-" + ups);
            Assertions.assertTrue(bootstrap.cloud().getServiceInfos().isEmpty());

            try {
                bootstrap.serviceInfoByName(ups).timeout(Duration.ofMillis(1)).hasElements().block();
                Assertions.fail();
            } catch (Exception err) {
                Assertions.assertTrue(ExceptionUtils.getRootCause(err).getClass().equals(TimeoutException.class));
            }
        } finally {
            bootstrap.shutdown();
        }
    }

    @Test
    public void plainServiceInfo() {
        PlainServiceInfo si1 = new PlainServiceInfo("key", "https://a.b/c?d=e");
        PlainServiceInfo si2 = new PlainServiceInfo("key", "https://a.b/c?d=e1&d=e2");
        PlainServiceInfo si3 = new PlainServiceInfo("key", "https://a.b?d=e");

        Flux<PlainServiceInfo> flux = Flux
                .fromIterable(Arrays.asList(si1, si1, si1, si2, si3))
                .log()
                .distinctUntilChanged(Function.identity(), UPSs::isEquals)
                .distinctUntilChanged(Function.identity(), (s1, s2) -> UPSs.hashCode(s1) == UPSs.hashCode(s2));

        AtomicLong l = new AtomicLong();
        flux.subscribe(new Consumer<PlainServiceInfo>() {
            @Override
            public void accept(PlainServiceInfo t) {
                logger.info("accepting UPS: {}", t);
                l.incrementAndGet();
            }
        });

        flux.subscribe();
        Assertions.assertEquals(3, l.get());
    }

    @Test
    public void rawServiceInfo() {
        RawServiceInfo si1 = new RawServiceInfo("key", "a".getBytes());
        RawServiceInfo si2 = new RawServiceInfo("key", "b".getBytes());
        RawServiceInfo si3 = new RawServiceInfo("key", "c".getBytes());

        Flux<RawServiceInfo> flux = Flux
                .fromIterable(Arrays.asList(si1, si1, si1, si2, si2, si3))
                .log()
                .distinctUntilChanged(Function.identity(), UPSs::isEquals)
                .distinctUntilChanged(Function.identity(), (s1, s2) -> UPSs.hashCode(s1) == UPSs.hashCode(s2));

        AtomicLong l = new AtomicLong();
        flux.subscribe(new Consumer<RawServiceInfo>() {
            @Override
            public void accept(RawServiceInfo t) {
                logger.info("accepting UPS: {}", t.getId());
                l.incrementAndGet();
            }
        });

        flux.subscribe();
        Assertions.assertEquals(3, l.get());
    }

    @Test
    public void pg() {
        PostgresqlServiceInfo si1 = new PostgresqlServiceInfo("key", "postgres://app_owner:app_owner@127.0.0.1:5432/defaultdb1");
        PostgresqlServiceInfo si2 = new PostgresqlServiceInfo("key", "postgres://app_owner:app_owner2@127.0.0.1:5432/defaultdb");
        PostgresqlServiceInfo si3 = new PostgresqlServiceInfo("key", "postgres://app_owner:app_owner@127.0.0.3:5432/defaultdb");

        Flux<PostgresqlServiceInfo> flux = Flux
                .fromIterable(Arrays.asList(si1, si1, si1, si2, si2, si3))
                .log()
                .distinctUntilChanged(Function.identity(), UPSs::isEquals)
                .distinctUntilChanged(Function.identity(), (s1, s2) -> UPSs.hashCode(s1) == UPSs.hashCode(s2));

        AtomicLong l = new AtomicLong();
        flux.subscribe(new Consumer<PostgresqlServiceInfo>() {
            @Override
            public void accept(PostgresqlServiceInfo t) {
                logger.info("accepting UPS: {}", t.getId());
                l.incrementAndGet();
            }
        });

        flux.subscribe();
        Assertions.assertEquals(3, l.get());
    }
}
