package org.springframework.cloud;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.BiConsumer;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableObject;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.InitCommand;
import org.eclipse.jgit.api.PullCommand;
import org.eclipse.jgit.errors.TransportException;
import org.eclipse.jgit.lib.ConfigConstants;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.RemoteConfig;
import org.eclipse.jgit.transport.TagOpt;
import org.eclipse.jgit.transport.URIish;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.service.ServiceInfo;

import com.netflix.archaius.config.DefaultSettableConfig;
import com.turbospaces.boot.AbstractBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.CloudMapConfig;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.cfg.DynamicCompositeConfig;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.UPSs;

public class ConfigurableCloudFactory extends CloudFactory {
    private final Logger logger = LoggerFactory.getLogger( getClass() );

    private final AbstractBootstrap bootstrap;
    private final Cloud cloud;
    private final MutableObject<URL> logFile;

    public ConfigurableCloudFactory(ConfigurableCloudConnector connector, AbstractBootstrap bootstrap, MutableObject<URL> logFile) throws Exception {
        registerCloudConnector( connector );

        this.bootstrap = Objects.requireNonNull( bootstrap );
        this.logFile = Objects.requireNonNull( logFile );
        this.cloud = super.getCloud();

        // ~ schedule dynamic reload
        ApplicationConfig cfg = bootstrap.props().cfg();
        Optional<ServiceInfo> opt = UPSs.findServiceInfoByName( cloud, UPSs.CFG );
        if ( opt.isPresent() ) {
            if ( bootstrap.props().APP_CFG_DYNAMIC_POLLING_ENABLED.get() ) {
                // ~ periodically poll for changes
                cfg.execute( new Runnable() {
                    @Override
                    public void run() {
                        try {
                            apply();
                        }
                        catch ( TransportException ex ) {
                            logger.warn( ex.getMessage(), ex );
                        }
                        catch ( Exception ex ) {
                            logger.error( ex.getMessage(), ex );
                        }
                    }
                } );
            }
            else {
                apply();
            }
        }
        else {
            logger.debug( "'{}' is not bound to app ...", UPSs.CFG );
        }

        // ~ replace configuration with real implementation
        CloudMapConfig cfc = new CloudMapConfig( connector.getApplicationInstanceInfo() );
        DefaultSettableConfig config = (DefaultSettableConfig) cfg.getConfig( DynamicCompositeConfig.CLOUD_CFG_NAME );
        config.setProperties( cfc );
    }
    @Override
    public Cloud getCloud() {
        return cloud; // ~ safe to call many times
    }
    private void apply() throws Exception {
        ApplicationConfig cfg = bootstrap.props().cfg();
        Optional<ServiceInfo> opt = UPSs.findServiceInfoByName( cloud, UPSs.CFG );
        if ( opt.isPresent() ) {
            PlainServiceInfo info = (PlainServiceInfo) opt.get();
            DefaultSettableConfig gitCfg = (DefaultSettableConfig) cfg.getConfig( DynamicCompositeConfig.GIT_CFG_NAME );

            File toFile = Files.createTempDirectory( "cfg" ).toFile();
            InitCommand initCmd = new InitCommand().setBare( false ).setDirectory( toFile );
            logger.debug( "cloning git repo from={}", info );

            try (Git git = initCmd.call()) {
                applyConfig( git, info );

                // pull
                PullCommand pullCommand = git.pull().setRemote( Constants.DEFAULT_REMOTE_NAME ).setTagOpt( TagOpt.NO_TAGS );
                pullCommand.setRemoteBranchName( Constants.MASTER );
                if ( StringUtils.isNotEmpty( info.getUserName() ) && StringUtils.isNotEmpty( info.getPassword() ) ) {
                    pullCommand.setCredentialsProvider( new UsernamePasswordCredentialsProvider( info.getUserName(), info.getPassword() ) );
                }
                pullCommand.call();

                Map<String, Object> cloudProps = cloud.getApplicationInstanceInfo().getProperties();
                String spaceName = cloudProps.get( CloudOptions.CLOUD_APP_SPACE_NAME ).toString();
                String appId = cloud.getApplicationInstanceInfo().getAppId();
                String filename = "application.properties";

                Properties gitProps = new Properties();
                Runnable task = new Runnable() {
                    @Override
                    public void run() {
                        Set<String> toRemove = new HashSet<>();
                        gitCfg.forEachProperty( new BiConsumer<String, Object>() {
                            @Override
                            public void accept(String k, Object v) {
                                if ( gitProps.containsKey( k ) ) {}
                                else {
                                    toRemove.add( k );
                                }
                            }
                        } );
                        if ( toRemove.isEmpty() ) {}
                        else {
                            logger.debug( "found orphaned properties: {}", toRemove );
                            for ( String prop : toRemove ) {
                                gitCfg.clearProperty( prop );
                            }
                        }
                        logger.debug( "setting git cfg props={}", gitProps );
                        cfg.setGitProperties( gitProps );
                    }
                };

                logger.debug( "looking for '{}' in {}:{}", filename, spaceName, appId );
                File dspace = new File( toFile, spaceName );
                if ( dspace.exists() && dspace.isDirectory() ) {
                    File fspace = new File( dspace, filename );

                    // ~ try load space shared properties
                    if ( fspace.exists() && fspace.isFile() ) {
                        try (FileInputStream io = new FileInputStream( fspace )) {
                            gitProps.load( io );

                        }
                    }

                    File dapp = new File( dspace, appId );
                    if ( dapp.exists() && dspace.isDirectory() ) {
                        File fapp = new File( dapp, filename );
                        if ( fapp.exists() && fapp.isFile() ) {
                            try (FileInputStream io = new FileInputStream( fapp )) {
                                gitProps.load( io );
                            }
                        }

                        filename = "logging.xml";
                        fapp = new File( dapp, filename );
                        if ( fapp.exists() && fapp.isFile() ) {
                            logFile.setValue( fapp.toURI().toURL() );
                        }
                    }
                }

                task.run(); // ~ set properties
            }
            finally {
                if ( toFile.exists() ) {
                    toFile.delete();
                }
            }
        }
    }
    private void applyConfig(Git git, PlainServiceInfo info) throws URISyntaxException, IOException {
        logger.debug( "patching git config ..." );

        StoredConfig config = git.getRepository().getConfig();
        config.setBoolean( ConfigConstants.CONFIG_GC_SECTION, null, ConfigConstants.CONFIG_KEY_AUTODETACH, false );
        if ( bootstrap.isDevMode() ) {
            config.setBoolean( "http", null, "sslVerify", false );
        }

        URIish urish = new URIish( StringUtils.removeEnd( info.getUri(), "/" ) );
        RemoteConfig remoteConfig = new RemoteConfig( config, Constants.DEFAULT_REMOTE_NAME );
        remoteConfig.addURI( urish );

        RefSpec refSpec = new RefSpec();
        refSpec = refSpec.setForceUpdate( true );
        refSpec = refSpec.setSourceDestination( Constants.R_HEADS + "*", Constants.R_REMOTES + remoteConfig.getName() + "/" + "*" );
        remoteConfig.addFetchRefSpec( refSpec );
        remoteConfig.update( config );

        config.save();
    }
}
