package com.turbospaces.cfg;

import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.net.URL;
import java.time.Duration;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.function.Consumer;
import java.util.regex.Pattern;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.common.collect.Range;
import com.google.common.reflect.TypeToken;
import com.netflix.archaius.api.Property;

import reactor.core.scheduler.Scheduler;
import reactor.util.retry.Retry;

public interface TypedPropertyFactory {
    String APPLICATION_PROPERTIES = "application.properties";

    DynamicPropertyFactory factory();

    String externalIp();
    Retry retry(Scheduler scheduler);
    boolean warnInconsistency(String... cfgs) throws Exception;

    default ApplicationConfig cfg() {
        return factory().getConfig();
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<List<Integer>> listOfInts(String key) {
        TypeToken<List<Integer>> type = new TypeToken<List<Integer>>() {};
        Class<List<Integer>> rawType = (Class<List<Integer>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<List<Long>> listOfLongs(String key) {
        TypeToken<List<Long>> type = new TypeToken<List<Long>>() {};
        Class<List<Long>> rawType = (Class<List<Long>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<List<String>> listOfStrings(String key) {
        TypeToken<List<String>> type = new TypeToken<List<String>>() {};
        Class<List<String>> rawType = (Class<List<String>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<List<Double>> listOfDoubles(String key) {
        TypeToken<List<Double>> type = new TypeToken<List<Double>>() {};
        Class<List<Double>> rawType = (Class<List<Double>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<List<BigDecimal>> listOfBigDecimals(String key) {
        TypeToken<List<BigDecimal>> type = new TypeToken<List<BigDecimal>>() {};
        Class<List<BigDecimal>> rawType = (Class<List<BigDecimal>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Set<LocalDate>> setOfLocalDates(String key) {
        TypeToken<Set<LocalDate>> type = new TypeToken<Set<LocalDate>>() {};
        Class<Set<LocalDate>> rawType = (Class<Set<LocalDate>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Set<Pattern>> setOfPatterns(String key) {
        TypeToken<Set<Pattern>> type = new TypeToken<Set<Pattern>>() {};
        Class<Set<Pattern>> rawType = (Class<Set<Pattern>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Set<Integer>> setOfInts(String key) {
        TypeToken<Set<Integer>> type = new TypeToken<Set<Integer>>() {};
        Class<Set<Integer>> rawType = (Class<Set<Integer>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Set<Long>> setOfLongs(String key) {
        TypeToken<Set<Long>> type = new TypeToken<Set<Long>>() {};
        Class<Set<Long>> rawType = (Class<Set<Long>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Set<String>> setOfStrings(String key) {
        TypeToken<Set<String>> type = new TypeToken<Set<String>>() {};
        Class<Set<String>> rawType = (Class<Set<String>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Set<Double>> setOfDoubles(String key) {
        TypeToken<Set<Double>> type = new TypeToken<Set<Double>>() {};
        Class<Set<Double>> rawType = (Class<Set<Double>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Set<BigDecimal>> setOfBigDecimals(String key) {
        TypeToken<Set<BigDecimal>> type = new TypeToken<Set<BigDecimal>>() {};
        Class<Set<BigDecimal>> rawType = (Class<Set<BigDecimal>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<List<LocalDate>> listOfLocalDates(String key) {
        TypeToken<List<LocalDate>> type = new TypeToken<List<LocalDate>>() {};
        Class<List<LocalDate>> rawType = (Class<List<LocalDate>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<List<Pattern>> listOfPatterns(String key) {
        TypeToken<List<Pattern>> type = new TypeToken<List<Pattern>>() {};
        Class<List<Pattern>> rawType = (Class<List<Pattern>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast);
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Map<String, String>> mapOfStringStrings(String key) {
        TypeToken<Map<String, String>> type = new TypeToken<Map<String, String>>() {};
        Class<Map<String, String>> rawType = (Class<Map<String, String>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast).orElse(Collections.emptyMap());
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Map<String, Integer>> mapOfStringInts(String key) {
        TypeToken<Map<String, Integer>> type = new TypeToken<Map<String, Integer>>() {};
        Class<Map<String, Integer>> rawType = (Class<Map<String, Integer>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast).orElse(Collections.emptyMap());
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Map<String, Long>> mapOfStringLongs(String key) {
        TypeToken<Map<String, Long>> type = new TypeToken<Map<String, Long>>() {};
        Class<Map<String, Long>> rawType = (Class<Map<String, Long>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast).orElse(Collections.emptyMap());
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Map<String, Double>> mapOfStringDoubles(String key) {
        TypeToken<Map<String, Double>> type = new TypeToken<Map<String, Double>>() {};
        Class<Map<String, Double>> rawType = (Class<Map<String, Double>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast).orElse(Collections.emptyMap());
    }
    @SuppressWarnings({ "serial", "unchecked" })
    default Property<Map<String, BigDecimal>> mapOfStringBigDecimals(String key) {
        TypeToken<Map<String, BigDecimal>> type = new TypeToken<Map<String, BigDecimal>>() {};
        Class<Map<String, BigDecimal>> rawType = (Class<Map<String, BigDecimal>>) type.getRawType();
        return factory().get(key, type.getType()).map(rawType::cast).orElse(Collections.emptyMap());
    }
    default Property<Duration> rangeValue(String key, Duration defaultValue, Range<Duration> boundary) {
        Logger logger = LoggerFactory.getLogger(getClass());
        Property<Duration> prop = factory().get(key, Duration.class).orElse(defaultValue);
        if (Objects.nonNull(boundary)) {
            prop.subscribe(new Consumer<Duration>() {
                @Override
                public void accept(Duration t) {
                    if (Objects.nonNull(t)) {
                        if (boundary.contains(t)) {
                            logger.error("property: {} is out of range: {}, new value: {}", prop.getKey(), boundary, t);
                        }
                    }
                }
            });
            logger.debug("subscribed to: {} and will monitor value of range: {}", prop.getKey(), boundary);
        }
        return prop;
    }
    default void analyze(URL url) throws Exception {
        Map<String, Object> map = readFieldAsMap();

        Properties table = new Properties();
        try (InputStream io = url.openStream()) {
            table.load(io);
        }

        for (Entry<Object, Object> it : table.entrySet()) {
            String key = it.getKey().toString();
            String value = it.getValue().toString();
            if (BooleanUtils.isFalse(map.containsKey(key))) {
                System.err.println(String.format("unknown: %s", key));
            } else {
                String toString = Objects.toString(map.get(key));
                if (Objects.equals(toString, value)) {
                    System.out.println(String.format("toRemove: %s", key));
                }
            }
        }
    }
    default Map<String, Object> readFieldAsMap() throws IllegalAccessException {
        Map<String, Object> map = Maps.newHashMap();

        for (Field f : FieldUtils.getAllFields(getClass())) {
            if (Modifier.isPublic(f.getModifiers())) {
                if (BooleanUtils.isFalse(Modifier.isStatic(f.getModifiers()))) {
                    Property<?> prop = (Property<?>) FieldUtils.readField(f, this);
                    String key = prop.getKey();
                    Object value = prop.get();
                    map.put(key, value);
                }
            }
        }

        //
        // ~ remove common props
        //
        Arrays.asList(FieldUtils.getAllFields(CloudOptions.class)).stream().map(Field::getName).forEach(map::remove);

        return map;
    }
    default void putValueIfPresent(String key, ImmutableMap.Builder<String, String> map) {
        if (factory().getConfig().containsKey(key)) {
            String raw = Objects.requireNonNull(factory().getConfig().getString(key));
            map.put(key, raw);
        }
    }
    default void putValue(String key, ImmutableMap.Builder<String, String> map, String defaultValue) {
        if (factory().getConfig().containsKey(key)) {
            String raw = Objects.requireNonNull(factory().getConfig().getString(key));
            map.put(key, raw);
        } else {
            map.put(key, Objects.requireNonNull(defaultValue));
        }
    }
}
