package com.turbospaces.boot;

import java.io.InputStream;
import java.net.URL;
import java.time.Duration;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang3.RandomStringUtils;
import org.eclipse.jgit.lib.Repository;
import org.springframework.cloud.SmartCloud;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.util.ResourceUtils;

import com.google.common.net.HostAndPort;
import com.netflix.archaius.api.exceptions.ConfigException;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.UPSs;

public class MockCloud implements SmartCloud {
    private final Properties cloudPropsOverride = new Properties();
    private final ApplicationConfig cfg;
    private final Set<ServiceInfo> services = new LinkedHashSet<>();
    private final String spaceName;
    private final String appId;
    private final String appName;

    private MockCloud(String appId, String appName) throws ConfigException {
        this.appId = Objects.requireNonNull(appId);
        this.appName = Objects.requireNonNull(appName);
        this.spaceName = RandomStringUtils.randomAlphabetic(4);
        this.cfg = ApplicationConfig.create(Duration.ofSeconds(30));
    }
    @Override
    public void addUps(ServiceInfo info) {
        services.add(info);
    }
    @Override
    public boolean removeUps(String id) {
        for (Iterator<ServiceInfo> it = services.iterator(); it.hasNext();) {
            ServiceInfo si = it.next();
            if (si.getId().equals(id)) {
                it.remove();
                return true;
            }
        }

        return false;
    }
    @Override
    public boolean removeUps(ServiceInfo si) {
        return services.remove(si);
    }
    public MockCloud witCfg(int port, Repository repo) {
        return withCfg(HostAndPort.fromParts("localhost", port), repo);
    }
    public MockCloud withCfg(HostAndPort addr, Repository repo) {
        PlainServiceInfo info = new PlainServiceInfo(UPSs.CFG, PlainServiceInfo.HTTP_SCHEME, addr, null, null, repo.getIdentifier());
        addUserProvidedService(info);
        return this;
    }
    public MockCloud appPort(int port) {
        cloudPropsOverride.put(CloudOptions.CLOUD_APP_PORT, port);
        return this;
    }
    public MockCloud addUserProvidedService(ServiceInfo info) {
        services.add(info);
        return this;
    }
    public ApplicationConfig build() throws Exception {
        String slot = "0";
        int port = PlatformUtil.findAvailableTcpPort();
        String hostname = PlatformUtil.detectIp();

        // mock typical cloud properties
        Properties props = new Properties();
        props.put(CloudOptions.CLOUD_APP_ID, appId);
        props.put(CloudOptions.CLOUD_APP_SPACE_NAME, spaceName);
        props.put(CloudOptions.CLOUD_APP_HOST, hostname);
        props.put(CloudOptions.CLOUD_APP_PORT, port);
        props.put(CloudOptions.CLOUD_APP_INSTANCE_INDEX, slot);
        props.put(CloudOptions.CLOUD_APP_NAME, appName);
        props.putAll(cloudPropsOverride);

        // override local properties
        cfg.setLocalProperties(UPSs.addUserProvideServices(props, services));

        //
        return cfg;
    }
    public static MockCloud fromClasspathAppProps() throws Exception {
        // load required properties file
        URL resource = ResourceUtils.getURL(ResourceUtils.CLASSPATH_URL_PREFIX + "application.properties");
        return fromClasspathAppProps(resource);
    }
    public static MockCloud fromClasspathAppProps(URL url) throws Exception {
        // load required properties file
        try (InputStream io = url.openStream()) {
            Properties props = new Properties();
            props.load(io);
            String appId = props.getProperty(CloudOptions.CLOUD_APP_ID);
            String appName = props.getProperty(CloudOptions.CLOUD_APP_NAME, randomAppName());
            return new MockCloud(appId, appName);
        }
    }
    public static MockCloud newMock() throws ConfigException {
        int length = MockCloud.class.getSimpleName().length();
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric(length);
        return new MockCloud(randomAppId, randomAppName());
    }
    private static String randomAppName() {
        int length = MockCloud.class.getSimpleName().length();
        LocalDate now = LocalDate.now(ZoneOffset.UTC);
        String randomGroup = "group-" + RandomStringUtils.randomAlphanumeric(length);
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric(length);
        String randomVersion = now.getYear() + "" + now.getMonthValue() + "" + now.getDayOfMonth();
        String randomAppName = String.format("%s:%s:%s", randomGroup, randomAppId, randomVersion);
        return randomAppName;
    }
}
