package com.turbospaces.common;

import java.util.Map;
import java.util.Objects;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiConsumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class CompletableRequestReplyMapper<K, V> implements RequestReplyMapper<K, V> {
    private final Logger logger = LoggerFactory.getLogger( getClass() );
    private final ConcurrentMap<K, CompletableFuture<V>> corr = new ConcurrentHashMap<>();
    private final Timer timer = new Timer( true );

    @Override
    public CompletableFuture<V> acquire(K key, int timeout, TimeUnit unit) {
        CompletableFuture<V> f = new CompletableFuture<>();
        long now = System.currentTimeMillis();
        f.whenComplete( new BiConsumer<V, Throwable>() {
            @Override
            public void accept(V resp, Throwable err) {
                if ( err != null ) {
                    CompletableFuture<?> promise = corr.remove( key );
                    if ( promise != null ) {
                        if ( err instanceof TimeoutException ) {
                            logger.debug( "request-reply(m={}) removing subj due to timeout", key );
                        }
                    }
                }
                else {
                    logger.debug( "request-reply(m={}) completed in {} 'ms'", key, ( System.currentTimeMillis() - now ) );
                }
            }
        } );
        boolean duplicate = corr.putIfAbsent( key, f ) != null;
        if ( duplicate ) {
            throw new RuntimeException( "duplicate key violation corrId: " + key );
        }

        Map<String, String> mdc = MDC.getCopyOfContextMap(); // ~ capture MDC
        timer.schedule( new TimerTask() {
            @Override
            public void run() {
                if ( f.isDone() ) {}
                else {
                    if ( mdc != null ) {
                        MDC.setContextMap( mdc ); // ~ set MDC stuff back if necessary
                    }

                    try {
                        String msg = String.format( "no response in %s%s for %s", timeout, unit.toString().toLowerCase(), key );
                        TimeoutException timeoutException = new TimeoutException( msg );
                        f.completeExceptionally( timeoutException );
                    }
                    finally {
                        MDC.clear(); // ~ finally clear it if necessary
                    }
                }
            }
        }, unit.toMillis( timeout ) );
        return f;
    }
    @Override
    public void complete(K key, V value) {
        CompletableFuture<V> subj = corr.remove( Objects.requireNonNull( key ) );
        if ( subj != null ) {
            subj.complete( value );
        }
        else {
            logger.trace( "no such correlation for key: {}", key );
        }
    }
    @Override
    public void completeExceptionally(K key, Throwable reason) {
        CompletableFuture<V> subj = corr.remove( Objects.requireNonNull( key ) );
        if ( subj != null ) {
            subj.completeExceptionally( reason );
        }
    }
    @Override
    public void completeExceptionally(Throwable reason) {
        for ( CompletableFuture<V> f : corr.values() ) {
            f.completeExceptionally( reason );
        }
    }
    @Override
    public void clear() {
        corr.clear();
    }
    @Override
    public int pendingCount() {
        return corr.size();
    }
}
