package org.springframework.cloud;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.springframework.cloud.service.ServiceConnectorCreator;
import org.springframework.cloud.service.ServiceInfo;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;

public class DynamicCloud extends Cloud implements SmartCloud {
    private final Map<String, ServiceInfo> upss = Maps.newConcurrentMap();

    public DynamicCloud(CloudConnector cloudConnector, List<ServiceConnectorCreator<?, ? extends ServiceInfo>> serviceConnectorCreators) {
        super(cloudConnector, serviceConnectorCreators);
    }
    @Override
    public List<ServiceInfo> getServiceInfos() {
        if (upss.isEmpty()) {
            return super.getServiceInfos(); // ~ optimization
        }

        ImmutableList.Builder<ServiceInfo> l = ImmutableList.builder();
        l.addAll(super.getServiceInfos());
        l.addAll(upss.values());

        return l.build();
    }
    @Override
    public ServiceInfo getServiceInfo(String serviceId) {
        ServiceInfo si = upss.get(serviceId);
        return Objects.nonNull(si) ? si : super.getServiceInfo(serviceId);
    }
    @SuppressWarnings({ "unchecked" })
    @Override
    public <T extends ServiceInfo> List<T> getServiceInfosByType(Class<T> serviceInfoType) {
        ImmutableList.Builder<T> l = ImmutableList.builder();
        l.addAll(super.getServiceInfosByType(serviceInfoType));

        for (ServiceInfo serviceInfo : upss.values()) {
            if (serviceInfoType.isAssignableFrom(serviceInfo.getClass())) {
                l.add((T) serviceInfo);
            }
        }

        return l.build();
    }
    @Override
    public void addUps(ServiceInfo info) {
        List<ServiceInfo> l = super.getServiceInfos();
        for (ServiceInfo si : l) {
            if (si.getId().equals(info.getId())) {
                throw new IllegalArgumentException(String.format("ups: %s already exists and defined in cloud connector", info.getId()));
            }
        }

        //
        // ~ register override
        //
        upss.put(info.getId(), info);
    }
    @Override
    public boolean removeUps(String id) {
        return Objects.nonNull(upss.remove(id));
    }
    @Override
    public boolean removeUps(ServiceInfo si) {
        return Objects.nonNull(upss.remove(si.getId()));
    }
}
