package org.springframework.cloud;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.BiConsumer;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableObject;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.InitCommand;
import org.eclipse.jgit.api.PullCommand;
import org.eclipse.jgit.lib.ConfigConstants;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.RemoteConfig;
import org.eclipse.jgit.transport.TagOpt;
import org.eclipse.jgit.transport.URIish;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

import com.netflix.archaius.config.DefaultCompositeConfig;
import com.netflix.archaius.config.DefaultSettableConfig;
import com.turbospaces.boot.AbstractBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.CloudMapConfig;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.cfg.DynamicCompositeConfig;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.UPSs;

public class ConfigurableCloudFactory extends CloudFactory {
    private static final String[] PROPERTY_FILE_EXTENSIONS = new String[] { "properties", "props" };

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final ThreadPoolTaskScheduler timer = new ThreadPoolTaskScheduler();
    private final Object mutex = new Object();
    private final AbstractBootstrap<?> bootstrap;
    private final DynamicCloud cloud;
    private final MutableObject<URL> logFile;

    public ConfigurableCloudFactory(SmartCloudConnector connector, AbstractBootstrap<?> bootstrap, MutableObject<URL> logFile) throws Exception {
        registerCloudConnector(connector);

        this.bootstrap = Objects.requireNonNull(bootstrap);
        this.logFile = Objects.requireNonNull(logFile);

        //
        // ~ if matching cloud
        //
        if (connector.isInMatchingCloud()) {
            this.cloud = new DynamicCloud(connector, getServiceCreators());
        }
        else {
            throw new CloudException("No suitable cloud connector found");
        }

        timer.setDaemon(true);
        timer.afterPropertiesSet();

        Optional<ServiceInfo> opt = UPSs.findServiceInfoByName(cloud, UPSs.CFG);
        if (opt.isPresent()) {
            //
            // ~ trigger immediately on current thread
            //
            refresh();

            //
            // ~ periodically poll for changes
            //
            if (bootstrap.props().APP_CFG_DYNAMIC_POLLING_ENABLED.get()) {
                timer.scheduleWithFixedDelay(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            refresh();
                        } catch (Throwable err) {
                            logger.error(err.getMessage(), err);
                        }
                    }
                }, bootstrap.props().APP_TIMER_INTERVAL.get());
            }
        } else {
            logger.debug("'{}' is not bound to app ...", UPSs.CFG);
        }

        //
        // ~ replace configuration with real implementation
        //
        CloudMapConfig cfc = new CloudMapConfig(connector.getApplicationInstanceInfo());
        DefaultSettableConfig config = (DefaultSettableConfig) bootstrap.cfg().getConfig(DynamicCompositeConfig.CLOUD_CFG_NAME);
        config.setProperties(cfc);
    }
    @Override
    public DynamicCloud getCloud() {
        return cloud; // ~ safe to call many times
    }
    public void refresh() throws Exception {
        synchronized (mutex) {
            Optional<ServiceInfo> opt = UPSs.findServiceInfoByName(cloud, UPSs.CFG);
            if (opt.isPresent()) {
                PlainServiceInfo info = (PlainServiceInfo) opt.get();

                File toFile = Files.createTempDirectory("cfg").toFile();
                InitCommand initCmd = new InitCommand().setBare(false).setDirectory(toFile);
                logger.debug("cloning git repo from={}", info);

                try (Git git = initCmd.call()) {
                    applyConfig(git, info);

                    // pull
                    PullCommand pullCommand = git.pull().setRemote(Constants.DEFAULT_REMOTE_NAME).setTagOpt(TagOpt.NO_TAGS);
                    pullCommand.setRemoteBranchName(Constants.MASTER);
                    if (StringUtils.isNotEmpty(info.getUserName()) && StringUtils.isNotEmpty(info.getPassword())) {
                        pullCommand.setCredentialsProvider(new UsernamePasswordCredentialsProvider(info.getUserName(), info.getPassword()));
                    }
                    pullCommand.call();

                    Map<String, Object> cloudProps = cloud.getApplicationInstanceInfo().getProperties();
                    String spaceName = cloudProps.get(CloudOptions.CLOUD_APP_SPACE_NAME).toString();
                    String appId = cloud.getApplicationInstanceInfo().getAppId();

                    logger.debug("looking for property files in {}:{}", spaceName, appId);
                    File dspace = new File(toFile, spaceName);
                    if (dspace.exists() && dspace.isDirectory()) {
                        //
                        // ~ try load space shared properties
                        //
                        Collection<File> dprops = FileUtils.listFiles(dspace, PROPERTY_FILE_EXTENSIONS, false);
                        if (Objects.nonNull(dprops)) {
                            for (File f : dprops) {
                                if (f.isFile()) {
                                    load(f);
                                }
                            }
                        }

                        File dapp = new File(dspace, appId);
                        if (dapp.exists() && dapp.isDirectory()) {
                            //
                            // ~ try load application properties
                            //
                            Collection<File> aprops = FileUtils.listFiles(dapp, PROPERTY_FILE_EXTENSIONS, false);
                            if (Objects.nonNull(aprops)) {
                                for (File f : aprops) {
                                    if (f.isFile()) {
                                        load(f);
                                    }
                                }
                            }

                            String filename = "logging.xml";
                            File fapp = new File(dapp, filename);
                            if (fapp.exists() && fapp.isFile()) {
                                logFile.setValue(fapp.toURI().toURL());
                            }
                        }
                    }
                } finally {
                    if (toFile.exists()) {
                        toFile.delete();
                    }
                }
            }
        }
    }
    private void load(File f) throws Exception {
        ApplicationConfig cfg = bootstrap.cfg();

        Properties props = new Properties();
        try (FileInputStream io = new FileInputStream(f)) {
            logger.debug("loading space shared propertries from: {}", f.getAbsolutePath());
            props.load(io);
        }

        String name = Paths.get(f.getParentFile().getName(), f.getName()).toString();

        DefaultCompositeConfig gitComposeConfig = (DefaultCompositeConfig) cfg.getConfig(DynamicCompositeConfig.GIT_CFG_NAME);
        DefaultSettableConfig fcfg = (DefaultSettableConfig) gitComposeConfig.getConfig(name);
        if (Objects.isNull(fcfg)) {
            fcfg = new DefaultSettableConfig();
            gitComposeConfig.addConfig(name, fcfg);
        }

        Set<String> toRemove = new HashSet<>();
        fcfg.forEachProperty(new BiConsumer<String, Object>() {
            @Override
            public void accept(String k, Object v) {
                if (props.containsKey(k)) {

                } else {
                    toRemove.add(k);
                }
            }
        });

        if (toRemove.isEmpty()) {} else {
            logger.debug("found orphaned properties: {}", toRemove);
            for (String prop : toRemove) {
                fcfg.clearProperty(prop);
            }
        }

        logger.debug("setting git cfg props={}", props);
        fcfg.setProperties(props);
    }
    private void applyConfig(Git git, PlainServiceInfo info) throws URISyntaxException, IOException {
        logger.debug("patching git config ...");

        StoredConfig config = git.getRepository().getConfig();
        config.setBoolean(ConfigConstants.CONFIG_GC_SECTION, null, ConfigConstants.CONFIG_KEY_AUTODETACH, false);
        if (bootstrap.isDevMode()) {
            config.setBoolean("http", null, "sslVerify", false);
        }

        URIish urish = new URIish(StringUtils.removeEnd(info.getUri(), "/"));
        RemoteConfig remoteConfig = new RemoteConfig(config, Constants.DEFAULT_REMOTE_NAME);
        remoteConfig.addURI(urish);

        RefSpec refSpec = new RefSpec();
        refSpec = refSpec.setForceUpdate(true);
        refSpec = refSpec.setSourceDestination(Constants.R_HEADS + "*", Constants.R_REMOTES + remoteConfig.getName() + "/" + "*");
        remoteConfig.addFetchRefSpec(refSpec);
        remoteConfig.update(config);

        config.save();
    }
}
