package com.turbospaces.boot;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.time.Duration;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Properties;

import org.apache.commons.lang3.RandomStringUtils;
import org.eclipse.jgit.lib.Repository;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.common.RedisServiceInfo;
import org.springframework.util.ResourceUtils;

import com.google.common.io.ByteSource;
import com.google.common.net.HostAndPort;
import com.netflix.archaius.api.exceptions.ConfigException;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.ups.H2ServiceInfo;
import com.turbospaces.ups.KafkaServiceInfo;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.RawServiceInfo;
import com.turbospaces.ups.UPSs;

public class MockCloud {
    private final Properties cloudPropsOverride = new Properties();
    private final ApplicationConfig cfg;
    private final List<ServiceInfo> services = new LinkedList<>();
    private final String spaceName;
    private final String appId;
    private final String appName;

    private MockCloud(String appId, String appName) throws ConfigException {
        this.appId = Objects.requireNonNull(appId);
        this.appName = Objects.requireNonNull(appName);
        this.spaceName = RandomStringUtils.randomAlphabetic(4);
        this.cfg = ApplicationConfig.create(Duration.ofSeconds(30));
    }
    public MockCloud withKafka(int port) {
        return withKafka(HostAndPort.fromParts("localhost", port));
    }
    public MockCloud withKafka(HostAndPort addr) {
        KafkaServiceInfo info = new KafkaServiceInfo(addr);
        addUserProvidedService(info);
        return this;
    }
    public MockCloud withRedis(int port) {
        return withRedis(HostAndPort.fromParts("localhost", port));
    }
    public MockCloud withRedis(HostAndPort addr) {
        RedisServiceInfo info = new RedisServiceInfo(UPSs.REDIS, addr.getHost(), addr.getPort(), "na");
        addUserProvidedService(info);
        return this;
    }
    public MockCloud witCfg(int port, Repository repo) {
        return withCfg(HostAndPort.fromParts("localhost", port), repo);
    }
    public MockCloud withCfg(HostAndPort addr, Repository repo) {
        PlainServiceInfo info = new PlainServiceInfo(UPSs.CFG, PlainServiceInfo.HTTP_SCHEME, addr, null, null, repo.getIdentifier());
        addUserProvidedService(info);
        return this;
    }
    public MockCloud withRawService(String id, URL url) throws IOException {
        try (InputStream io = url.openStream()) {
            RawServiceInfo serviceInfo = new RawServiceInfo(id, new ByteSource() {
                @Override
                public InputStream openStream() throws IOException {
                    return io;
                }
            }.read());
            addUserProvidedService(serviceInfo);
            return this;
        }
    }
    public MockCloud withH2(boolean durable) {
        return withH2(durable, spaceName);
    }
    public MockCloud withH2(boolean durable, String namespace) {
        String url = "mem:" + namespace;
        if (durable) {
            url += ";" + "DB_CLOSE_DELAY=-1";
        }
        H2ServiceInfo ownerInfo = new H2ServiceInfo(UPSs.H2_OWNER, url);
        H2ServiceInfo appInfo = new H2ServiceInfo(UPSs.H2_APP, url);
        addUserProvidedService(ownerInfo);
        addUserProvidedService(appInfo);
        return this;
    }
    public MockCloud appPort(int port) {
        cloudPropsOverride.put(CloudOptions.CLOUD_APP_PORT, port);
        return this;
    }
    public MockCloud addUserProvidedService(ServiceInfo info) {
        services.add(info);
        return this;
    }
    public ApplicationConfig build() throws Exception {
        String slot = "0";
        int port = PlatformUtil.findAvailableTcpPort();
        String hostname = PlatformUtil.detectIp();

        // mock typical cloud properties
        Properties props = new Properties();
        props.put(CloudOptions.CLOUD_APP_ID, appId);
        props.put(CloudOptions.CLOUD_APP_SPACE_NAME, spaceName);
        props.put(CloudOptions.CLOUD_APP_HOST, hostname);
        props.put(CloudOptions.CLOUD_APP_PORT, port);
        props.put(CloudOptions.CLOUD_APP_INSTANCE_INDEX, slot);
        props.put(CloudOptions.CLOUD_APP_NAME, appName);
        props.putAll(cloudPropsOverride);

        // override local properties
        cfg.setLocalProperties(UPSs.addUserProvideServices(props, services));

        //
        return cfg;
    }
    public static MockCloud fromClasspathAppProps() throws Exception {
        // load required properties file
        URL resource = ResourceUtils.getURL(ResourceUtils.CLASSPATH_URL_PREFIX + "application.properties");
        return fromClasspathAppProps(resource);
    }
    public static MockCloud fromClasspathAppProps(URL url) throws Exception {
        // load required properties file
        try (InputStream io = url.openStream()) {
            Properties props = new Properties();
            props.load(io);
            String appId = props.getProperty(CloudOptions.CLOUD_APP_ID);
            String appName = props.getProperty(CloudOptions.CLOUD_APP_NAME, randomAppName());
            return new MockCloud(appId, appName);
        }
    }
    public static MockCloud newMock() throws ConfigException {
        int length = MockCloud.class.getSimpleName().length();
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric(length);
        return new MockCloud(randomAppId, randomAppName());
    }
    private static String randomAppName() {
        int length = MockCloud.class.getSimpleName().length();
        LocalDate now = LocalDate.now(ZoneOffset.UTC);
        String randomGroup = "group-" + RandomStringUtils.randomAlphanumeric(length);
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric(length);
        String randomVersion = now.getYear() + "" + now.getMonthValue() + "" + now.getDayOfMonth();
        String randomAppName = String.format("%s:%s:%s", randomGroup, randomAppId, randomVersion);
        return randomAppName;
    }
}
