package com.turbospaces.boot;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.google.common.base.Throwables;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.SSL;
import com.turbospaces.common.SelfSignedCertificateGenerator;
import com.turbospaces.common.ThrowableAction1;
import io.micrometer.core.instrument.MeterRegistry;
import io.opentracing.Tracer;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.mutable.MutableObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.cloud.Cloud;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.support.GenericApplicationContext;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.Schedulers;
import reactor.util.retry.Retry;

import javax.net.ssl.SSLContext;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.lang.reflect.UndeclaredThrowableException;
import java.security.KeyStore;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;

public interface Bootstrap extends BeanFactory {
    //
    // life-cycle
    //
    ConfigurableApplicationContext run(String... args);

    void shutdown() throws Exception; // ~ graceful shutdown

    void exit(int code); // ~ exit VM manually

    Cloud cloud();

    String spaceName();

    String appId();

    String release();

    boolean isDevMode();

    GenericApplicationContext context();

    Platform globalPlatform();

    ExecutionPlatform platform(String name);

    ApplicationProperties props();

    ApplicationConfig cfg();

    KeyStore keyStore();

    default void createSelfSignedKeyStoreIfAbsent(File file) throws Exception {
        createSelfSignedKeyStoreIfAbsent(file, SelfSignedCertificateGenerator.PASSWORD);
    }

    default void createSelfSignedKeyStoreIfAbsent(File file, String password) throws Exception {
        if (file.exists()) {
        } else {
            SelfSignedCertificateGenerator ssc = new SelfSignedCertificateGenerator();
            ssc.setBootstrap(this);
            KeyStore keystore = ssc.call();
            try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
                keystore.store(out, password.toCharArray());
                FileUtils.writeByteArrayToFile(file, out.toByteArray());
            }
        }
    }

    default Optional<SSLContext> generateSelfSignedCertificate() throws Exception {
        if (props().APP_USE_SELF_SIGNED_CERTIFICATE.get()) {
            // command to export cert from ca, may be helpful when testing smth in safari locally
            // keytool -export -keystore frontend_keystore -alias uam-frontend -file frontend.cer
            if (isDevMode()) {
                File keystoreFile = new File(FileUtils.getUserDirectory().getAbsolutePath() + "/self_signed_keystore");
                createSelfSignedKeyStoreIfAbsent(keystoreFile);

                SSL ssl = new SSL();
                ssl.loadKeyStore(keystoreFile, SelfSignedCertificateGenerator.PASSWORD);
                return Optional.of(ssl.build());
            }
        }
        return Optional.empty();
    }

    void refreshCfg() throws Exception;

    void squashLogging() throws Exception;

    //
    // ~ port
    //
    int port();

    int secondaryPort();

    int tertiaryPort();

    //
    // health-checks and metrics support
    //
    MetricRegistry metricRegistry();

    MeterRegistry meterRegistry();

    HealthCheckRegistry healthCheckRegistry();

    Tracer tracer();

    //
    // ~
    //
    boolean addChannel(Channel acceptor);

    //
    // helper-methods
    //
    boolean isHealthy();

    void registerHealthCheck(String name, HealthCheck check);

    default Retry retry() {
        return props().retry(globalPlatform().scheduler());
    }

    default <V> void retry(V value, ThrowableAction1<V> action) {
        retry(value, action, globalPlatform().scheduler());
    }

    default <V> void retry(V value, ThrowableAction1<V> action, Scheduler scheduler) {
        Logger logger = LoggerFactory.getLogger(getClass());
        AtomicLong seq = new AtomicLong();

        Mono.just(value).doOnError(new Consumer<Throwable>() {
            @Override
            public void accept(Throwable t) {
                logger.error(t.getMessage(), t);
            }
        }).doOnNext(new Consumer<V>() {
            @Override
            public void accept(V target) {
                try {
                    action.apply(target);
                } catch (Exception err) {
                    logger.warn("got failure on iteration: " + seq.getAndIncrement(), err);
                    Throwables.throwIfUnchecked(err);
                    throw new UndeclaredThrowableException(err);
                }
            }
        }).retryWhen(props().retry(scheduler)).block();
    }

    default <V> void retry(Flux<V> stream, ThrowableAction1<List<V>> action) {
        retry(stream, action, globalPlatform().scheduler());
    }

    default <V> void retry(Flux<V> stream, ThrowableAction1<List<V>> action, Scheduler scheduler) {
        Logger logger = LoggerFactory.getLogger(getClass());
        AtomicLong seq = new AtomicLong();

        stream.collectList().doOnError(new Consumer<Throwable>() {
            @Override
            public void accept(Throwable t) {
                logger.error(t.getMessage(), t);
            }
        }).doOnNext(new Consumer<List<V>>() {
            @Override
            public void accept(List<V> target) {
                try {
                    action.apply(target);
                } catch (Exception err) {
                    logger.warn("got failure on iteration: " + seq.getAndIncrement(), err);
                    Throwables.throwIfUnchecked(err);
                    throw new UndeclaredThrowableException(err);
                }
            }
        }).retryWhen(props().retry(scheduler)).block();
    }

    default <V> V retryCallable(Callable<V> callback) {
        return retryCallable(callback, Schedulers.immediate());
    }

    default <V> V retryCallable(Callable<V> callback, Scheduler scheduler) {
        Logger logger = LoggerFactory.getLogger(getClass());
        MutableObject<V> result = new MutableObject<V>();
        AtomicLong seq = new AtomicLong();

        Mono.just(callback).doOnError(new Consumer<Throwable>() {
            @Override
            public void accept(Throwable t) {
                logger.error(t.getMessage(), t);
            }
        }).doOnNext(new Consumer<Callable<V>>() {
            @Override
            public void accept(Callable<V> target) {
                try {
                    V value = target.call();
                    result.setValue(value);
                } catch (Exception err) {
                    logger.warn("got failure on iteration: " + seq.getAndIncrement(), err);
                    Throwables.throwIfUnchecked(err);
                    throw new UndeclaredThrowableException(err);
                }
            }
        }).retryWhen(props().retry(scheduler)).block();

        return result.getValue();
    }
}
