package com.turbospaces.boot;

import java.io.IOException;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;

import javax.inject.Inject;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.bootstrap.HttpServer;
import org.apache.http.impl.bootstrap.ServerBootstrap;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.http.protocol.HttpRequestHandlerMapper;
import org.apache.http.util.EntityUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.AbstractFactoryBean;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.BootstrapContextClosedEvent;
import org.springframework.boot.BootstrapRegistry;
import org.springframework.boot.BootstrapRegistry.InstanceSupplier;
import org.springframework.boot.BootstrapRegistryInitializer;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.scheduling.annotation.Scheduled;

import com.codahale.metrics.health.HealthCheck;
import com.netflix.archaius.api.annotations.Configuration;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.common.ThrowableAction;
import com.turbospaces.healthchecks.HttpHealthCheck;
import com.turbospaces.healthchecks.SocketHealthCheck;
import com.turbospaces.ups.PlainServiceInfo;

import io.jaegertracing.client.Version;

public class BootstrapTest {
    @Test
    public void works() throws Throwable {
        int port = PlatformUtil.findAvailableTcpPort();

        MockCloud cfg = MockCloud.newMock();
        cfg.appPort(port);
        cfg.withKafka(9092);

        URIBuilder uri = new URIBuilder().setScheme("http").setHost("localhost").setPath("/v1/status").setPort(port);

        ApplicationProperties props = new ApplicationProperties(cfg.build());

        SimpleBootstrap bootstrap = new SimpleBootstrap(AppConfig.class, props);
        bootstrap.addBootstrapRegistryInitializer(new HttpClientPlugin());
        bootstrap.addChannel(new InboundChannel(uri.build()));
        bootstrap.run();

        try {
            HealthCheck check1 = new HttpHealthCheck(props, uri.build());
            Assertions.assertTrue(check1.execute().isHealthy());
            SocketHealthCheck check2 = new SocketHealthCheck(props, new PlainServiceInfo("my-service-" + Version.get(), uri.toString()));
            Assertions.assertTrue(check2.execute().isHealthy());
        } finally {
            bootstrap.shutdown();
        }
    }

    @Configuration
    @Import(SchedulingDiModule.class)
    public static class AppConfig {
        Logger logger = LoggerFactory.getLogger(getClass());

        @Bean
        public HttpFactory factory(HttpClientBuilder client) {
            return new HttpFactory(client);
        }
        @Bean
        public Foo foo(HttpFactory factory) throws Exception {
            return new Foo(factory.getObject());
        }
        @Scheduled(fixedRate = 1000)
        public void mock() {
            logger.info("The time is now {}", new SimpleDateFormat("HH:mm:ss").format(new Date()));
        }
    }

    public static class InboundChannel extends AbstractBootstrapAware implements Channel {
        private final URI uri;
        private ServerBootstrap http = ServerBootstrap.bootstrap();
        private HttpServer channel;

        @Inject
        CloseableHttpClient httpClient;

        public InboundChannel(URI uri) {
            this.uri = uri;
        }
        @Override
        public void run(ApplicationArguments args) throws Exception {
            http = ServerBootstrap.bootstrap();
            http.setListenerPort(bootstrap.port());
            http.setHandlerMapper(new HttpRequestHandlerMapper() {
                @Override
                public HttpRequestHandler lookup(HttpRequest request) {
                    return new HttpRequestHandler() {
                        @Override
                        public void handle(HttpRequest req, HttpResponse resp, HttpContext context) throws HttpException, IOException {
                            String reqUri = req.getRequestLine().getUri();
                            if (reqUri.equals(uri.getPath())) {
                                resp.setStatusCode(HttpStatus.SC_OK);
                                resp.setEntity(new StringEntity(req.toString(), ContentType.APPLICATION_JSON));
                            } else {
                                resp.setStatusCode(HttpStatus.SC_NOT_FOUND);
                            }
                        }
                    };
                }
            });
            channel = http.create();
            channel.start();

            HttpGet req = new HttpGet();
            req.setURI(uri);

            CountDownLatch latch = new CountDownLatch(1);
            try (CloseableHttpResponse resp = httpClient.execute(req)) {
                String text = EntityUtils.toString(resp.getEntity());
                bootstrap.globalPlatform().work(new ThrowableAction() {
                    @Override
                    public void apply() throws Exception {
                        logger.debug(text);
                        latch.countDown();
                    }
                });
            }
            latch.await();
        }
        @Override
        public void destroy() {
            if (channel != null) {
                channel.stop();
            }
        }
    }

    public static class Foo implements BootstrapAware {
        private Bootstrap bootstrap;

        @Value(CloudOptions.CLOUD_APP_ID)
        String appId;

        private final CloseableHttpClient client;

        @Inject
        public Foo(CloseableHttpClient client) {
            this.client = Objects.requireNonNull(client);
        }
        @Override
        public void setBootstrap(Bootstrap bootstrap) {
            this.bootstrap = bootstrap;
        }
        @Override
        public String toString() {
            ToStringBuilder toString = PlatformUtil.noClassNameToString(this);
            return toString.append("appId", appId).append("client", client).append("version", bootstrap.release()).build();
        }
    }

    public static class HttpClientPlugin implements BootstrapRegistryInitializer {
        @Override
        public void initialize(BootstrapRegistry registry) {
            registry.register(HttpClientBuilder.class, InstanceSupplier.of(HttpClients.custom()));
            registry.addCloseListener(new ApplicationListener<BootstrapContextClosedEvent>() {
                @Override
                public void onApplicationEvent(BootstrapContextClosedEvent event) {
                    HttpClientBuilder builder = registry.getRegisteredInstanceSupplier(HttpClientBuilder.class).get(event.getBootstrapContext());

                    ConfigurableApplicationContext ctx = event.getApplicationContext();
                    ctx.getBeanFactory().registerSingleton("http-client-factory", builder);
                }
            });
        }
    }

    public static class HttpFactory extends AbstractFactoryBean<CloseableHttpClient> {
        private final HttpClientBuilder builder;

        public HttpFactory(HttpClientBuilder builder) {
            this.builder = Objects.requireNonNull(builder);
        }
        @Override
        public Class<?> getObjectType() {
            return CloseableHttpClient.class;
        }
        @Override
        protected CloseableHttpClient createInstance() throws Exception {
            return builder.build();
        }
        @Override
        protected void destroyInstance(CloseableHttpClient instance) throws Exception {
            instance.close();
        }
    }
}
