package com.turbospaces.cfg;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.net.URL;
import java.time.Duration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.ConfigurableCloudConnector;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.ResourceUtils;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Primitives;
import com.netflix.archaius.DefaultDecoder;
import com.netflix.archaius.api.TypeConverter;
import com.netflix.archaius.api.TypeConverter.Factory;
import com.netflix.archaius.api.TypeConverter.Registry;
import com.netflix.archaius.api.config.SettableConfig;
import com.netflix.archaius.api.exceptions.ConfigException;
import com.netflix.archaius.config.DefaultCompositeConfig;
import com.netflix.archaius.config.DefaultSettableConfig;
import com.netflix.archaius.config.EmptyConfig;
import com.netflix.archaius.config.MapConfig;
import com.netflix.archaius.config.SystemConfig;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import com.typesafe.config.ConfigSyntax;
import com.typesafe.config.ConfigValue;

public class ApplicationConfig extends DefaultCompositeConfig implements DynamicCompositeConfig, CloudOptions {
    private final Logger logger = LoggerFactory.getLogger(ApplicationConfig.class);

    private final DynamicPropertyFactory propertyFactory;

    public ApplicationConfig(Duration refreshInterval) throws ConfigException {
        //
        // ~ no need to add sensitive properties
        //
        List<String> ignorePattern = new LinkedList<>();
        ignorePattern.add(ConfigurableCloudConnector.ENV_UPS_PREFIX);
        ignorePattern.add(ConfigurableCloudConnector.ENV_SPACE_NAME);
        ignorePattern.add(ConfigurableCloudConnector.ENV_HOSTNAME);

        // replace
        Map<String, String> oldgetenv = new HashMap<>(System.getenv());
        Map<String, String> newgetenv = new HashMap<>();
        for (Entry<String, String> it : oldgetenv.entrySet()) {
            String oldProp = it.getKey();
            String newProp = oldProp.toLowerCase().replaceAll("_", ".");

            boolean allowed = true;
            for (String s : ignorePattern) {
                if (oldProp.startsWith(s)) {
                    allowed = false;
                    break;
                }
            }
            if (allowed) {
                logger.trace("adding ENV prop {}={}", oldProp, newProp, it.getValue());
                newgetenv.put(newProp, it.getValue());
            }
        }

        //
        // ~ ordered hierarchy
        //
        addConfig(RUNTIME_CFG_NAME, new DefaultSettableConfig()); // runtime
        addConfig(LOCAL_CFG_NAME, new DefaultSettableConfig()); // local-dev-%username%
        addConfig(CLOUD_CFG_NAME, new DefaultSettableConfig()); // CFG -> replace later
        addConfig(CMD_LINE_CFG_NAME, new DefaultSettableConfig()); // CLI arguments
        addConfig(SYSTEM_ENV_CFG_NAME, new MapConfig(newgetenv)); // converted ENV variables
        addConfig(SYSTEM_PROPS_CFG_NAME, new SystemConfig());
        addConfig(GIT_CFG_NAME, new DefaultCompositeConfig()); // GIT
        addConfig(LEGACY_CFG_NAME, EmptyConfig.INSTANCE);
        addConfig(DEFAULT_APP_CFG_NAME, new DefaultSettableConfig()); // default classpath
        addBuildMetaConfig(new DefaultSettableConfig()); // adding maven git commit id plugin generated props

        propertyFactory = DynamicPropertyFactory.from(this);

        try {
            setDecoder(decoder());
        } catch (IllegalArgumentException | IllegalAccessException err) {
            throw new ConfigException(err.getMessage(), err);
        }
    }
    @Override
    public DynamicPropertyFactory factory() {
        return propertyFactory;
    }
    @Override
    public void clearLocalProperty(String key) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(LOCAL_CFG_NAME);
        c.clearProperty(key);
        logger.info("clearing local prop: {}", key);
    }
    @Override
    public void setLocalProperty(String key, Object value) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(LOCAL_CFG_NAME);
        c.setProperty(key, value);
        logger.info("setting local prop: {} -> {}", key, value);
    }
    @Override
    public void setLocalProperties(Properties props) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(LOCAL_CFG_NAME);
        c.setProperties(props);
        logger.info("setting local props: {}", props);
    }
    @Override
    public void clearCmdLineParams(String key) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(CMD_LINE_CFG_NAME);
        c.clearProperty(key);
        logger.info("clearing command line prop: {}", key);
    }
    @Override
    public void setCmdLineParams(String key, Object value) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(CMD_LINE_CFG_NAME);
        c.setProperty(key, value);
        logger.info("setting command line prop: {} -> {}", key, value);
    }
    @Override
    public void setCmdLineParams(Properties props) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(CMD_LINE_CFG_NAME);
        c.setProperties(props);
        logger.info("setting command line props: {}", props);
    }
    @Override
    public void clearDefaultProperty(String key) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(DEFAULT_APP_CFG_NAME);
        c.clearProperty(key);
        logger.info("clearing default prop: {}", key);
    }
    @Override
    public void setDefaultProperty(String key, Object value) {
        SettableConfig c = (SettableConfig) getConfig(DEFAULT_APP_CFG_NAME);
        c.setProperty(key, value);
        logger.info("setting default prop: {} -> {}", key, value);
    }
    @Override
    public void setDefaultProperties(Properties props) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig(DEFAULT_APP_CFG_NAME);
        c.setProperties(props);
        logger.info("setting default props: {}", props);
    }
    @Override
    public void clearAllDefaultProperties() throws ConfigException {
        logger.info("clearing default cfg: {}", ImmutableList.copyOf(getConfig(DEFAULT_APP_CFG_NAME).getKeys()));
        replaceConfig(DEFAULT_APP_CFG_NAME, new DefaultSettableConfig());
    }
    @Override
    public ApplicationConfig loadLocalDevProperties(String username) throws Exception {
        String userOverrideProps = String.format(ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-%s.properties", username);
        String userOverrideCfg = String.format(ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-%s.conf", username);

        URL propsURL = null;
        URL cfgURL = null;

        try {
            propsURL = ResourceUtils.getURL(userOverrideProps);
        } catch (FileNotFoundException err) {
            try {
                userOverrideProps = ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-template.properties";
                logger.debug("no user specific props file found, will try to lookup = '{}'", userOverrideProps);
                propsURL = ResourceUtils.getURL(userOverrideProps);
            } catch (FileNotFoundException io) {

            }
        }

        try {
            cfgURL = ResourceUtils.getURL(userOverrideCfg);
        } catch (FileNotFoundException err) {
            try {
                userOverrideCfg = ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-template.conf";
                logger.debug("no user specific cfg file found, attempting to lookup = '{}'", userOverrideCfg);
                cfgURL = ResourceUtils.getURL(userOverrideCfg);
            } catch (FileNotFoundException io) {

            }
        }

        if (propsURL != null) {
            try (InputStream io = propsURL.openStream()) {
                Properties tmp = new Properties();
                tmp.load(io);
                setLocalProperties(tmp);
            }
        }

        if (cfgURL != null) {
            File file = new File(cfgURL.toURI());

            ConfigParseOptions parseOptions = ConfigParseOptions.defaults().setAllowMissing(false).setSyntax(ConfigSyntax.CONF);
            Config typesafeConfig = ConfigFactory.parseFile(file, parseOptions).resolve();

            for (Map.Entry<String, ConfigValue> entry : typesafeConfig.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue().unwrapped();

                if (Objects.isNull(value)) {
                    clearLocalProperty(key);
                } else {
                    if (value instanceof List) {
                        List<?> list = (List<?>) value;
                        setLocalProperty(key, Joiner.on(getListDelimiter()).join(list));
                    } else if (value instanceof Set) {
                        Set<?> set = (Set<?>) value;
                        setLocalProperty(key, Joiner.on(getListDelimiter()).join(set));
                    } else if (value instanceof Map) {
                        Map<?, ?> map = (Map<?, ?>) value;
                        StringBuilder builer = new StringBuilder();
                        for (Iterator<?> it = map.entrySet().iterator(); it.hasNext();) {
                            Map.Entry<?, ?> next = (Entry<?, ?>) it.next();
                            builer.append(next.toString() + "=" + next.toString());
                            if (it.hasNext()) {
                                builer.append(getListDelimiter());
                            }
                        }
                        setLocalProperty(key, builer);
                    } else if (Primitives.allWrapperTypes().contains(value.getClass())) {
                        setLocalProperty(key, value);
                    } else {
                        setLocalProperty(key, value);
                    }
                }
            }
        }

        return this;
    }
    @Override
    public ApplicationConfig loadDefaultPropsFromResource(URL resource) throws IOException {
        try (InputStream io = resource.openStream()) {
            Properties props = new Properties();
            props.load(io);
            setDefaultProperties(props);
        }
        return this;
    }
    @Override
    public ApplicationConfig loadLocalDevProperties() throws Exception {
        String username = System.getProperty("user.name");
        loadLocalDevProperties(username);
        return this;
    }
    private void addBuildMetaConfig(DefaultSettableConfig cfg) throws ConfigException {
        try (InputStream is = this.getClass().getClassLoader().getResourceAsStream("git.properties")) {
            if (is != null) {
                addConfig(DynamicCompositeConfig.BUILD_META_CFG_NAME, cfg);
                Properties props = new Properties();
                props.load(is);
                cfg.setProperties(props);
            }
        } catch (IOException e) {
            logger.error("Can't load git build properties", e);
        }
    }
    @SuppressWarnings("unchecked")
    private DefaultDecoder decoder() throws IllegalArgumentException, IllegalAccessException {
        DefaultDecoder decoder = DefaultDecoder.INSTANCE;

        logger.info("attempting to patch default codec: {}", decoder);
        Field f = ReflectionUtils.findField(DefaultDecoder.class, "factories");
        f.setAccessible(true);

        List<TypeConverter.Factory> factories = (List<Factory>) f.get(decoder);

        factories.add(new Factory() {
            @Override
            public Optional<TypeConverter<?>> get(Type type, Registry registry) {
                if (type.equals(Pattern.class)) {
                    return Optional.of(Pattern::compile);
                }
                return Optional.empty();
            }
        });

        return decoder;
    }
    public static ApplicationConfig create() throws ConfigException {
        return create(Duration.ofSeconds(15));
    }
    public static ApplicationConfig create(Duration refreshInterval) throws ConfigException {
        return new ApplicationConfig(refreshInterval);
    }
    public static ApplicationConfig create(String appId, Duration refreshInterval) throws ConfigException {
        ApplicationConfig cfg = create();
        cfg.setDefaultProperty(CloudOptions.CLOUD_APP_ID, appId);
        return cfg;
    }
}
