package com.turbospaces.cfg;

import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Range;
import com.google.common.net.HttpHeaders;
import com.google.common.reflect.TypeToken;
import com.netflix.archaius.api.Property;
import com.netflix.archaius.config.DefaultCompositeConfig;
import com.netflix.archaius.config.DefaultSettableConfig;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.logging.ClassUtils;

import reactor.core.scheduler.Scheduler;
import reactor.util.retry.Retry;

public class ApplicationProperties implements TypedPropertyFactory {
    private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationProperties.class);

    public static final String APPLICATION_PROPERTIES = "application.properties";

    public static final int PRIMARY_PORT = 8089;
    public static final int SECONDARY_PORT = 8091;
    public static final int TERTIARY_PORT = 8093;

    public ApplicationProperties(ApplicationConfig cfg) {
        this.cfg = Objects.requireNonNull(cfg);
        DynamicPropertyFactory pf = cfg.factory();

        //
        // cloud
        //
        CLOUD_APP_ID = pf.get(CloudOptions.CLOUD_APP_ID, String.class).orElse(System.getProperty("user.name"));
        CLOUD_APP_INSTANCE_INDEX = pf.get(CloudOptions.CLOUD_APP_INSTANCE_INDEX, String.class).orElse("0");
        CLOUD_APP_NAME = pf.get(CloudOptions.CLOUD_APP_NAME, String.class);
        CLOUD_APP_SPACE_NAME = pf.get(CloudOptions.CLOUD_APP_SPACE_NAME, String.class);
        CLOUD_APP_HOST = pf.get(CloudOptions.CLOUD_APP_HOST, String.class).orElse("localhost");
        CLOUD_APP_PORT = pf.get(CloudOptions.CLOUD_APP_PORT, int.class).orElse(PRIMARY_PORT);
        CLOUD_APP_SECONDARY_PORT = pf.get(CloudOptions.CLOUD_APP_SECONDARY_PORT, int.class).orElse(SECONDARY_PORT);
        CLOUD_APP_TERTIARY_PORT = pf.get(CloudOptions.CLOUD_APP_TERTIARY_PORT, int.class).orElse(TERTIARY_PORT);

        //
        // ~ CACHE
        //
        CACHE_DEFAULT_MAX_TTL = pf.get("cache.default.max-ttl", Duration.class).orElse(Duration.ofMinutes(60));
        CACHE_DEFAULT_MAX_IDLE = pf.get("cache.default.max-idle", Duration.class).orElse(Duration.ofMinutes(15));
        CACHE_DEFAULT_MAX_SIZE = pf.get("cache.default.max-size", int.class).orElse(1024 * 8);
        CACHE_REPLICATED_NEVER_EXPIRE = pf.get("cache.replicated.never-expire", boolean.class).orElse(true);
        CACHE_LOCAL_NEVER_EXPIRE = pf.get("cache.local.never-expire", boolean.class).orElse(false);

        //
        // TCP
        //
        TCP_REUSE_ADDRESS = pf.get("tcp.reuse-address", boolean.class).orElse(true);
        TCP_NO_DELAY = pf.get("tcp.no-delay", boolean.class).orElse(true);
        TCP_KEEP_ALIVE = pf.get("tcp.keep-alive", boolean.class).orElse(true);
        TCP_KEEP_ALIVE_TIMEOUT = rangeValue("tcp.keep-alive.timeout", Duration.ofMinutes(1), Range.closed(Duration.ofSeconds(0), Duration.ofMinutes(15)));
        TCP_CONNECTION_TIMEOUT = rangeValue("tcp.connection.timeout", Duration.ofSeconds(15), Range.closed(Duration.ofSeconds(0), Duration.ofMinutes(1)));
        TCP_SOCKET_TIMEOUT = rangeValue("tcp.socket.timeout", Duration.ofSeconds(60), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        TCP_SOCKET_BACKLOG = pf.get("tcp.socket.backlog", int.class).orElse(1024);
        TCP_FRAME_MAX_SIZE = pf.get("tcp.frame.max-size", int.class).orElse(1024 * 1024 * 16); // 16MB

        //
        // ~ NETTY
        //
        NETTY_ACCEPTOR_POOL_SIZE = pf.get("netty.acceptor-pool.size", int.class).orElse(Runtime.getRuntime().availableProcessors());
        NETTY_WORKER_POOL_SIZE = pf.get("netty.worker-pool.size", int.class).orElse(32);
        NETTY_VERSION_SEND_HTTP_HEADER = pf.get("netty.version.send-http-header", boolean.class).orElse(false);
        NETTY_VERSION_HTTP_HEADER_NAME = pf.get("netty.version.http-header-name", String.class).orElse("X-Netty-Version");

        //
        // ~ JETTY
        //
        JETTY_POOL_MIN_SIZE = pf.get("jetty.pool.min-size", int.class).orElse(32);
        JETTY_POOL_MAX_SIZE = pf.get("jetty.pool.max-size", int.class).orElse(64);
        JETTY_POOL_QUEUE_MAX_SIZE = pf.get("jetty.pool.queue-max-size", int.class).orElse(16 * 1024);
        JETTY_POOL_MAX_IDLE = pf.get("jetty.pool.max-idle", Duration.class).orElse(Duration.ofMinutes(1));
        JETTY_HTTP_SEND_DATE_HEADER = pf.get("jetty.http.send-date-header", boolean.class).orElse(true);
        JETTY_GZIP_ENABLED = pf.get("jetty.gzip.enabled", boolean.class).orElse(true);

        //
        // ~ CORS
        //
        // @formatter:off
        CORS_ALLOWED_ORIGINS = listOfStrings("cors.allowed-origins");
        CORS_ALLOWED_HEADERS = listOfStrings("cors.allowed-headers").orElse(List.of("access-control-allow-origin", "content-type", "sentry-trace", "x-platform"));
        CORS_ALLOWED_METHODS = listOfStrings("cors.allowed-methods").orElse(List.of("GET", "POST", "DELETE", "PUT", "PATCH"));
        CORS_EXPOSED_HEADERS = listOfStrings("cors.exposed-headers").orElse(List.of("location"));
        CORS_ALLOW_CREDENTIALS = pf.get("cors.allow-credentials", boolean.class).orElse(true);
        CORS_MAX_AGE  = rangeValue("cors.max-age", Duration.ofMinutes(1), Range.closed(Duration.ofSeconds(15), Duration.ofHours(1)));
        // @formatter:on

        //
        // ~ JDBC
        //
        JDBC_PERSIST_BATCH_SIZE = pf.get("jdbc.persist-batch.size", int.class).orElse(100);
        JDBC_QUERY_BATCH_SIZE = pf.get("jdbc.query-batch.size", int.class).orElse(250);
        JDBC_LAZY_BATCH_SIZE = pf.get("jdbc.lazy-batch.size", int.class).orElse(100);
        JDBC_LAZY_SEQUENCE_BATCH_SIZE = pf.get("jdbc.lazy-sequence-batch.size", int.class).orElse(1000);
        JDBC_POOL_MIN_SIZE = pf.get("jdbc.pool.min-size", int.class).orElse(1);
        JDBC_POOL_MAX_SIZE = pf.get("jdbc.pool.max-size", int.class).orElse(10);
        JDBC_READ_ONLY_POOL_MIN_SIZE = pf.get("jdbc.read-only-pool.min-size", int.class).orElse(1);
        JDBC_READ_ONLY_POOL_MAX_SIZE = pf.get("jdbc.read-only-pool.max-size", int.class).orElse(5);
        JDBC_CONNECTION_TIMEOUT = rangeValue("jdbc.connection.timeout", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(1)));

        //
        // ~ KAFKA
        //
        KAFKA_MAX_BLOCK = rangeValue("kafka.max-block", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(1)));
        KAFKA_IDEMPOTENCE_ENABLE = pf.get("kafka.idempotence.enable", boolean.class).orElse(true);
        KAFKA_ACKS = pf.get("kafka.acks", String.class).orElse("all");
        KAFKA_COMPRESSION_TYPE = pf.get("kafka.compression.type", String.class).orElse("none");
        KAFKA_RECORD_MAX_REQUEST_SIZE = pf.get("kafka.record.max-request-size", int.class).orElse(4 * 1024 * 1024); // 4 MB
        KAFKA_AUTO_OFFSET_RESET = pf.get("kafka.auto.offset.reset", String.class);
        KAFKA_HEARTBEAT_INTERVAL = rangeValue("kafka.heartbeat.interval", Duration.ofSeconds(5), Range.closed(Duration.ofSeconds(1), Duration.ofSeconds(15)));
        KAFKA_SESSION_TIMEOUT = rangeValue("kafka.session.timeout", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(1)));
        KAFKA_POLL_MAX_INTERVAL = rangeValue("kafka.poll.max-interval", Duration.ofMinutes(10), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(30)));
        KAFKA_POLL_MAX_RECORDS = pf.get("kafka.poll.max-records", int.class).orElse(1024);
        KAFKA_MIN_WORKERS = pf.get("kafka.min-workers", int.class).orElse(16);
        KAFKA_MAX_WORKERS = pf.get("kafka.max-workers", int.class).orElse(128);
        KAFKA_MAX_POLL_CONCURRENCY = pf.get("kafka.max-poll-concurrency", int.class).orElse(4);
        KAFKA_NACK_ON_QUEUE_FULL = pf.get("kafka.nack.on-queue-full", boolean.class).orElse(false);
        KAFKA_SYSTEM_EXIT_ON_QUEUE_FULL = pf.get("kafka.system-exit.on-queue-full", boolean.class).orElse(true);
        KAFKA_SYSTEM_EXIT_CODE = pf.get("kafka.system-exit.code", int.class).orElse(-1);
        KAFKA_SYSTEM_EXIT_DELAY = rangeValue("kafka.system-exit.delay", Duration.ofSeconds(5), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));

        //
        // ~ QUARTZ
        QUARTZ_SCHEDULER_ID = pf.get("quartz.scheduler.id", String.class).orElse("AUTO");
        QUARTZ_WORKER_POOL_COUNT = pf.get("quartz.worker-pool.count", Integer.class).orElse(8);
        QUARTZ_JOBSTORE_TABLE_PREFIX = pf.get("quartz.jobstore.table-prefix", String.class).orElse("qrtz_");
        QUARTZ_JOBSTORE_USE_PROPS = pf.get("quartz.jobstore.use-props", Boolean.class).orElse(false);
        QUARTZ_JOBSTORE_ACQUIRE_TRIGGERS_WITHIN_LOCK = pf.get("quartz.jobstore.acquire-triggers-within-lock", Boolean.class).orElse(false);
        QUARTZ_JOBSTORE_IS_CLUSTERED = pf.get("quartz.jobstore.is-clustered", Boolean.class).orElse(false);
        QUARTZ_CONNECTION_POOL_MIN = pf.get("quartz.connection-pool.min", Integer.class).orElse(1);
        QUARTZ_CONNECTION_POOL_MAX = pf.get("quartz.connection-pool.max", Integer.class).orElse(5);
        QUARTZ_INSTANCE_ID = pf.get("quartz.instance.id", String.class);
        QUARTZ_AUTO_RECOVERY_INTERVAL = pf.get("quartz.auto-recovery.interval", Duration.class).orElse(Duration.ofHours(1));
        QUARTZ_SHUTDOWN_WAIT_FOR_JOBS_COMPLETION = pf.get("quartz.shutdown.wait-for-jobs-completion", boolean.class);

        //
        // APP
        //
        APP_DEV_MODE = pf.get("app.dev.mode", boolean.class).orElse(true);
        APP_DNS_QUERY = pf.get("app.dns.query", String.class);
        APP_BACKOFF_RETRY_FIRST = rangeValue("app.backoff-retry.first", Duration.ofSeconds(1), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_BACKOFF_RETRY_MAX = rangeValue("app.backoff-retry.max", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(1)));
        APP_BACKOFF_RETRY_NUM = pf.get("app.backoff-retry.num", int.class).orElse(10);
        APP_SENTRY_ENABLED = pf.get("app.sentry.enabled", boolean.class).orElse(true);
        APP_USE_SELF_SIGNED_CERTIFICATE = pf.get("app.use.self-signed-certificate", boolean.class).orElse(false);
        APP_LOGGING_RESET_TO = pf.get("app.logging.reset-to", String.class);
        APP_LOGGING_DRY_RUN = pf.get("app.logging.dry-run", boolean.class).orElse(false);
        APP_METRICS_DRY_RUN = pf.get("app.metrics.dry-run", boolean.class).orElse(false);
        APP_METRICS_ELK_REPORTER_ENABLED = pf.get("app.metrics.elk-reporter.enabled", boolean.class).orElse(false);
        APP_METRICS_INFLUX_REPORTER_ENABLED = pf.get("app.metrics.influx-reporter.enabled", boolean.class).orElse(true);
        APP_METRICS_PROMETHEUS_REPORTER_ENABLED = pf.get("app.metrics.prometheus-reporter.enabled", boolean.class).orElse(true);
        APP_ALERTS_DRY_RUN = pf.get("app.alerts.dry-run", boolean.class).orElse(false);
        APP_EXTERNAL_IP = pf.get("app.external.ip", String.class).orElse(PlatformUtil.fetchExternalIp(this));
        APP_JMX_DOMAIN = pf.get("app.jmx.domain", String.class).orElse("metrics");
        APP_DB_MIGRATION_ENABLED = pf.get("app.db-migration.enabled", boolean.class).orElse(true);
        APP_DB_MIGRATION_PATH = pf.get("app.db-migration.path", String.class).orElse("db/sql-migration");
        APP_PLATFORM_POOL_SIZE = pf.get("app.platform.pool-size", int.class).orElse(64);
        APP_PLATFORM_MAX_IDLE = rangeValue("app.platform.max-idle", Duration.ofMinutes(1), Range.closed(Duration.ofSeconds(1), Duration.ofHours(1)));
        APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT = rangeValue("app.platform.graceful-shutdown-timeout", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_TIMER_INTERVAL = rangeValue("app.timer.interval", Duration.ofSeconds(15), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_METRICS_BULK_SIZE = pf.get("app.metrics.bulk-size", int.class).orElse(1024);
        APP_HEAP_DUMP_ON_OOM_ENABLED = pf.get("app.heap-dump-on-oom.enabled", boolean.class).orElse(true);
        APP_HEAP_DUMP_DIR = pf.get("app.heap-dump.dir", String.class).orElse(System.getProperty("java.io.tmpdir"));
        APP_WAIT_FOR_HEALTHCHECKS_INTERVAL = rangeValue("app.wait-for-healthchecks.interval", Duration.ofSeconds(1),Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_WAIT_FOR_HEALTHCHECKS_ENABLED = pf.get("app.wait-for-healthchecks.enabled", boolean.class).orElse(false);
        APP_WAIT_FOR_HEALTHCHECKS_TIMEOUT = rangeValue("app.wait-for-healthchecks.timeout", Duration.ofSeconds(60), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(5)));
        APP_DATA_START_CLEAN = pf.get("app.data.start-clean", boolean.class).orElse(false);
        APP_SHUTDOWN_HOOK_ENABLED = pf.get("app.shutdown-hook.enabled", boolean.class).orElse(true);
        APP_CLEAR_CONFIG_AT_SHUTDOWN_ENABLED = pf.get("app.clear-config-at-shutdown.enabled", boolean.class).orElse(false);
        APP_CFG_DYNAMIC_POLLING_ENABLED = pf.get("app.cfg-dynamic-polling.enabled", boolean.class).orElse(true);

        //
        // ~ HTTP
        //
        HTTP_POOL_MAX_PER_ROUTE = pf.get("http.pool.max-per-route", int.class).orElse(32);
        HTTP_POOL_MAX_SIZE = pf.get("http.pool.max-size", int.class).orElse(256);
        HTTP_METRICS_LATENCY_KEY = pf.get("http.metrics.latency-key", String.class).orElse("server-latency");
        HTTP_HEADERS_TO_MASK = listOfStrings("http.headers.to-mask").orElse(List.of(HttpHeaders.AUTHORIZATION, HttpHeaders.SET_COOKIE, "X-Api-Key"));
        HTTP_COOKIES_TO_MASK = listOfStrings("http.cookies.to-mask");
        HTTP_QUERY_PARAMS_TO_MASK = listOfStrings("http.query-params.to-mask").orElse(List.of("password"));
        HTTP_METRICS_INBOUND_PATH_MASK = listOfPatterns("http.metrics.inbound-path-mask");
        HTTP_METRICS_OUTBOUND_PATH_MASK = listOfPatterns("http.metrics.outbound-path-mask");
        HTTP_PROXY = pf.get("http.proxy", String.class);
        HTTP_REQUEST_TO_PROXY_PATTERNS = listOfPatterns("http.request-to-proxy.patterns");
        MOCK_HTTP_PROXY = pf.get("mock.http.proxy", String.class);
        MOCK_HTTP_REQUEST_TO_PROXY_PATTERNS = listOfPatterns("mock.http.request-to-proxy.patterns");

        //
        // ~ Sentry
        //
        SENTRY_ALERTS_FILTER_ENABLED = pf.get("sentry.alerts-filter.enabled", boolean.class).orElse(true);
        SENTRY_ALERTS_EX_CLASSES_TO_IGNORE = pf.getProperty("sentry.alerts.ex.classes.to.ignore").asType(s -> ClassUtils.parseThrowableClasses(s, ","), "");
        SENTRY_ALERTS_EX_MSGS_TO_IGNORE = pf.getProperty("sentry.alerts.ex.msgs.to.ignore").asType(s -> splitString(s, "~"), "");
        SENTRY_ALERTS_EX_PATTERNS_TO_IGNORE = pf.getProperty("sentry.alerts.ex.patterns.to.ignore").asType(s -> splitString(s, "~"), "");
        SENTRY_ALERTS_LOG_MESSAGES_TO_IGNORE = pf.getProperty("sentry.alerts.log.messages.to.ignore").asType(s -> splitString(s, "~"), "");
        SENTRY_ALERTS_LOGGERS_TO_IGNORE = pf.getProperty("sentry.alerts.loggers.to.ignore").asType(s -> splitString(s, "~"), "");
    }

    protected List<Pattern> parsePatterns(String patterns) {
        try {
            return patterns.isEmpty()
                    ? Collections.emptyList()
                    : Arrays.stream(patterns.split(",")).map(Pattern::compile).collect(Collectors.toList());
        } catch (Exception e) {
            LOGGER.error("Error on patter init {}", patterns, e);
            return Collections.emptyList();
        }
    }

    protected List<String> splitString(String value, String separator) {
        try {
            return value.trim().isEmpty() ? Collections.emptyList() : Arrays.asList(value.split(separator));
        } catch (Exception e) {
            LOGGER.error("Error on string list parse {}", value);
            return Collections.emptyList();
        }
    }
    @Override
    public ApplicationConfig cfg() {
        return cfg;
    }
    @Override
    public Property<Duration> rangeValue(String key, Duration defaultValue, Range<Duration> boundary) {
        Property<Duration> prop = cfg.factory().get(key, Duration.class).orElse(defaultValue);
        if (Objects.nonNull(boundary)) {
            prop.subscribe(new Consumer<Duration>() {
                @Override
                public void accept(Duration t) {
                    if (Objects.nonNull(t)) {
                        if (boundary.contains(t)) {
                            LOGGER.error("property: {} is out of range: {}, new value: {}", prop.getKey(), boundary, t);
                        }
                    }
                }
            });
            LOGGER.debug("subscribed to: {} and will monitor value of range: {}", prop.getKey(), boundary);
        }
        return prop;
    }
    @Override
    public Retry retry(Scheduler scheduler) {
        int numRetries = APP_BACKOFF_RETRY_NUM.get();
        Duration firstBackoff = APP_BACKOFF_RETRY_FIRST.get();
        Duration maxBackoff = APP_BACKOFF_RETRY_MAX.get();
        return Retry.backoff(numRetries, firstBackoff).maxBackoff(maxBackoff).scheduler(scheduler);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<List<Integer>> listOfInts(String key) {
        TypeToken<List<Integer>> type = new TypeToken<List<Integer>>() {};
        Class<List<Integer>> rawType = (Class<List<Integer>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<List<Long>> listOfLongs(String key) {
        TypeToken<List<Long>> type = new TypeToken<List<Long>>() {};
        Class<List<Long>> rawType = (Class<List<Long>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<List<String>> listOfStrings(String key) {
        TypeToken<List<String>> type = new TypeToken<List<String>>() {};
        Class<List<String>> rawType = (Class<List<String>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<List<Double>> listOfDoubles(String key) {
        TypeToken<List<Double>> type = new TypeToken<List<Double>>() {};
        Class<List<Double>> rawType = (Class<List<Double>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<List<BigDecimal>> listOfBigDecimals(String key) {
        TypeToken<List<BigDecimal>> type = new TypeToken<List<BigDecimal>>() {};
        Class<List<BigDecimal>> rawType = (Class<List<BigDecimal>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<Set<LocalDate>> setOfLocalDates(String key) {
        TypeToken<Set<LocalDate>> type = new TypeToken<Set<LocalDate>>() {};
        Class<Set<LocalDate>> rawType = (Class<Set<LocalDate>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<Set<Pattern>> setOfPatterns(String key) {
        TypeToken<Set<Pattern>> type = new TypeToken<Set<Pattern>>() {};
        Class<Set<Pattern>> rawType = (Class<Set<Pattern>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<Set<Integer>> setOfInts(String key) {
        TypeToken<Set<Integer>> type = new TypeToken<Set<Integer>>() {};
        Class<Set<Integer>> rawType = (Class<Set<Integer>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<Set<Long>> setOfLongs(String key) {
        TypeToken<Set<Long>> type = new TypeToken<Set<Long>>() {};
        Class<Set<Long>> rawType = (Class<Set<Long>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<Set<String>> setOfStrings(String key) {
        TypeToken<Set<String>> type = new TypeToken<Set<String>>() {};
        Class<Set<String>> rawType = (Class<Set<String>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<Set<Double>> setOfDoubles(String key) {
        TypeToken<Set<Double>> type = new TypeToken<Set<Double>>() {};
        Class<Set<Double>> rawType = (Class<Set<Double>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<Set<BigDecimal>> setOfBigDecimals(String key) {
        TypeToken<Set<BigDecimal>> type = new TypeToken<Set<BigDecimal>>() {};
        Class<Set<BigDecimal>> rawType = (Class<Set<BigDecimal>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<List<LocalDate>> listOfLocalDates(String key) {
        TypeToken<List<LocalDate>> type = new TypeToken<List<LocalDate>>() {};
        Class<List<LocalDate>> rawType = (Class<List<LocalDate>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    @SuppressWarnings({ "serial", "unchecked" })
    public Property<List<Pattern>> listOfPatterns(String key) {
        TypeToken<List<Pattern>> type = new TypeToken<List<Pattern>>() {};
        Class<List<Pattern>> rawType = (Class<List<Pattern>>) type.getRawType();
        return cfg.factory().get(key, type.getType()).map(rawType::cast);
    }
    @Override
    public boolean warnInconsistency(String... cfgs) throws Exception {
        boolean toReturn = false;

        ImmutableList<String> l = ArrayUtils.isEmpty(cfgs) ? ImmutableList.of(APPLICATION_PROPERTIES) : ImmutableList.copyOf(cfgs);
        DefaultCompositeConfig gitComposeConfig = (DefaultCompositeConfig) cfg.getConfig(DynamicCompositeConfig.GIT_CFG_NAME);
        Map<String, Object> map = readValues();

        for (String next : l) {
            Path path = Paths.get(CLOUD_APP_ID.get(), next);
            DefaultSettableConfig config = (DefaultSettableConfig) gitComposeConfig.getConfig(path.toString());

            if (Objects.nonNull(config)) {
                Iterator<String> it = config.getKeys();
                List<String> unknowns = Lists.newArrayList();

                while ( it.hasNext() ) {
                    String key = it.next();

                    if (BooleanUtils.isFalse(map.containsKey(key))) {
                        unknowns.add(key);
                        toReturn = true;
                    }
                }

                if (BooleanUtils.isFalse(unknowns.isEmpty())) {
                    LOGGER.error("unknown keys: {} in {}", unknowns, map.keySet());
                }
            }
        }

        return toReturn;
    }
    @Override
    public void analyze(URL url) throws Exception {
        Map<String, Object> map = readValues();

        Properties table = new Properties();
        try (InputStream io = url.openStream()) {
            table.load(io);
        }

        for (Entry<Object, Object> it : table.entrySet()) {
            String key = it.getKey().toString();
            String value = it.getValue().toString();
            if (BooleanUtils.isFalse(map.containsKey(key))) {
                System.err.println(String.format("unknown: %s", key));
            } else {
                String toString = Objects.toString(map.get(key));
                if (Objects.equals(toString, value)) {
                    System.out.println(String.format("toRemove: %s", key));
                }
            }
        }
    }
    protected Map<String, Object> readValues() throws IllegalAccessException {
        Map<String, Object> map = Maps.newHashMap();

        Field[] fields = FieldUtils.getAllFields(getClass());
        for (Field f : fields) {
            if (Modifier.isPublic(f.getModifiers()) && BooleanUtils.isFalse(Modifier.isStatic(f.getModifiers()))) {
                Property<?> prop = (Property<?>) FieldUtils.readField(f, this);
                String key = prop.getKey();
                Object value = prop.get();
                map.put(key, value);
            }
        }

        // ~ remove common props
        Arrays.asList(FieldUtils.getAllFields(CloudOptions.class)).stream().map(Field::getName).forEach(map::remove);

        return map;
    }

    //
    // ~ cloud
    //
    public final Property<String> CLOUD_APP_ID;
    public final Property<String> CLOUD_APP_SPACE_NAME;
    public final Property<String> CLOUD_APP_NAME;
    public final Property<String> CLOUD_APP_INSTANCE_INDEX;
    public final Property<String> CLOUD_APP_HOST;
    public final Property<Integer> CLOUD_APP_PORT;
    public final Property<Integer> CLOUD_APP_SECONDARY_PORT;
    public final Property<Integer> CLOUD_APP_TERTIARY_PORT;

    //
    // ~ CACHE
    //
    public final Property<Duration> CACHE_DEFAULT_MAX_TTL;
    public final Property<Duration> CACHE_DEFAULT_MAX_IDLE;
    public final Property<Integer> CACHE_DEFAULT_MAX_SIZE;
    public final Property<Boolean> CACHE_REPLICATED_NEVER_EXPIRE;
    public final Property<Boolean> CACHE_LOCAL_NEVER_EXPIRE;

    //
    // ~ TCP
    //
    public final Property<Boolean> TCP_REUSE_ADDRESS;
    public final Property<Boolean> TCP_NO_DELAY;
    public final Property<Boolean> TCP_KEEP_ALIVE;
    public final Property<Duration> TCP_KEEP_ALIVE_TIMEOUT;
    public final Property<Duration> TCP_CONNECTION_TIMEOUT;
    public final Property<Duration> TCP_SOCKET_TIMEOUT;
    public final Property<Integer> TCP_SOCKET_BACKLOG;
    public final Property<Integer> TCP_FRAME_MAX_SIZE;
    public final Property<Integer> HTTP_POOL_MAX_SIZE;

    //
    // ~ NETTY
    //
    public final Property<Integer> NETTY_ACCEPTOR_POOL_SIZE;
    public final Property<Integer> NETTY_WORKER_POOL_SIZE;
    public final Property<Boolean> NETTY_VERSION_SEND_HTTP_HEADER;
    public final Property<String> NETTY_VERSION_HTTP_HEADER_NAME;

    //
    // ~ JETTY
    //
    public final Property<Integer> JETTY_POOL_MAX_SIZE;
    public final Property<Integer> JETTY_POOL_MIN_SIZE;
    public final Property<Integer> JETTY_POOL_QUEUE_MAX_SIZE;
    public final Property<Duration> JETTY_POOL_MAX_IDLE;
    public final Property<Boolean> JETTY_HTTP_SEND_DATE_HEADER;
    public final Property<Boolean> JETTY_GZIP_ENABLED;

    //
    // ~ CORS
    //
    public final Property<List<String>> CORS_ALLOWED_ORIGINS;
    public final Property<List<String>> CORS_ALLOWED_HEADERS;
    public final Property<List<String>> CORS_ALLOWED_METHODS;
    public final Property<List<String>> CORS_EXPOSED_HEADERS;
    public final Property<Boolean> CORS_ALLOW_CREDENTIALS;
    public final Property<Duration> CORS_MAX_AGE;

    //
    // ~ JDBC
    //
    public final Property<Integer> JDBC_PERSIST_BATCH_SIZE;
    public final Property<Integer> JDBC_QUERY_BATCH_SIZE;
    public final Property<Integer> JDBC_LAZY_BATCH_SIZE;
    public final Property<Integer> JDBC_LAZY_SEQUENCE_BATCH_SIZE;
    public final Property<Integer> JDBC_POOL_MIN_SIZE;
    public final Property<Integer> JDBC_POOL_MAX_SIZE;
    public final Property<Integer> JDBC_READ_ONLY_POOL_MIN_SIZE;
    public final Property<Integer> JDBC_READ_ONLY_POOL_MAX_SIZE;
    public final Property<Duration> JDBC_CONNECTION_TIMEOUT;

    //
    // ~ KAFKA
    //
    public final Property<Duration> KAFKA_MAX_BLOCK;
    public final Property<Boolean> KAFKA_IDEMPOTENCE_ENABLE;
    public final Property<String> KAFKA_ACKS;
    public final Property<String> KAFKA_COMPRESSION_TYPE;
    public final Property<Integer> KAFKA_RECORD_MAX_REQUEST_SIZE;
    public final Property<String> KAFKA_AUTO_OFFSET_RESET;
    public final Property<Duration> KAFKA_HEARTBEAT_INTERVAL;
    public final Property<Duration> KAFKA_SESSION_TIMEOUT;
    public final Property<Duration> KAFKA_POLL_MAX_INTERVAL;
    public final Property<Integer> KAFKA_POLL_MAX_RECORDS;
    public final Property<Integer> KAFKA_MIN_WORKERS;
    public final Property<Integer> KAFKA_MAX_WORKERS;
    public final Property<Integer> KAFKA_MAX_POLL_CONCURRENCY;
    public final Property<Boolean> KAFKA_SYSTEM_EXIT_ON_QUEUE_FULL;
    public final Property<Boolean> KAFKA_NACK_ON_QUEUE_FULL;
    public final Property<Integer> KAFKA_SYSTEM_EXIT_CODE;
    public final Property<Duration> KAFKA_SYSTEM_EXIT_DELAY;

    //
    // ~ QUARTZ
    //
    public final Property<String> QUARTZ_SCHEDULER_ID;
    public final Property<Integer> QUARTZ_WORKER_POOL_COUNT;
    public final Property<String> QUARTZ_JOBSTORE_TABLE_PREFIX;
    public final Property<Boolean> QUARTZ_JOBSTORE_USE_PROPS;
    public final Property<Boolean> QUARTZ_JOBSTORE_ACQUIRE_TRIGGERS_WITHIN_LOCK;
    public final Property<Integer> QUARTZ_CONNECTION_POOL_MIN;
    public final Property<Integer> QUARTZ_CONNECTION_POOL_MAX;
    public final Property<Boolean> QUARTZ_JOBSTORE_IS_CLUSTERED;
    public final Property<String> QUARTZ_INSTANCE_ID;
    public final Property<Duration> QUARTZ_AUTO_RECOVERY_INTERVAL;
    public final Property<Boolean> QUARTZ_SHUTDOWN_WAIT_FOR_JOBS_COMPLETION;

    //
    // ~ HTTP
    //
    public final Property<List<Pattern>> HTTP_METRICS_INBOUND_PATH_MASK;
    public final Property<List<Pattern>> HTTP_METRICS_OUTBOUND_PATH_MASK;
    public final Property<String> HTTP_METRICS_LATENCY_KEY;
    public final Property<String> HTTP_PROXY;
    public final Property<Integer> HTTP_POOL_MAX_PER_ROUTE;
    public final Property<List<Pattern>> HTTP_REQUEST_TO_PROXY_PATTERNS;
    public final Property<String> MOCK_HTTP_PROXY;
    public final Property<List<Pattern>> MOCK_HTTP_REQUEST_TO_PROXY_PATTERNS;
    public final Property<List<String>> HTTP_HEADERS_TO_MASK;
    public final Property<List<String>> HTTP_QUERY_PARAMS_TO_MASK;
    public final Property<List<String>> HTTP_COOKIES_TO_MASK;

    //
    // ~ APP
    //
    public final Property<Boolean> APP_DEV_MODE;
    public final Property<String> APP_DNS_QUERY;
    public final Property<Duration> APP_BACKOFF_RETRY_FIRST;
    public final Property<Duration> APP_BACKOFF_RETRY_MAX;
    public final Property<Integer> APP_BACKOFF_RETRY_NUM;
    public final Property<Boolean> APP_SENTRY_ENABLED;
    public final Property<Boolean> APP_USE_SELF_SIGNED_CERTIFICATE;
    public final Property<String> APP_LOGGING_RESET_TO;
    public final Property<Boolean> APP_LOGGING_DRY_RUN;
    public final Property<Boolean> APP_METRICS_DRY_RUN;
    public final Property<Boolean> APP_METRICS_ELK_REPORTER_ENABLED;
    public final Property<Boolean> APP_METRICS_INFLUX_REPORTER_ENABLED;
    public final Property<Boolean> APP_METRICS_PROMETHEUS_REPORTER_ENABLED;
    public final Property<Boolean> APP_ALERTS_DRY_RUN;
    public final Property<String> APP_EXTERNAL_IP;
    public final Property<String> APP_JMX_DOMAIN;
    public final Property<Boolean> APP_DB_MIGRATION_ENABLED;
    public final Property<String> APP_DB_MIGRATION_PATH;
    public final Property<Integer> APP_PLATFORM_POOL_SIZE;
    public final Property<Duration> APP_PLATFORM_MAX_IDLE;
    public final Property<Duration> APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT;
    public final Property<Duration> APP_TIMER_INTERVAL;
    public final Property<Integer> APP_METRICS_BULK_SIZE;
    public final Property<Boolean> APP_HEAP_DUMP_ON_OOM_ENABLED;
    public final Property<String> APP_HEAP_DUMP_DIR;
    public final Property<Boolean> APP_DATA_START_CLEAN;
    public final Property<Duration> APP_WAIT_FOR_HEALTHCHECKS_INTERVAL;
    public final Property<Boolean> APP_WAIT_FOR_HEALTHCHECKS_ENABLED;
    public final Property<Duration> APP_WAIT_FOR_HEALTHCHECKS_TIMEOUT;
    public final Property<Boolean> APP_SHUTDOWN_HOOK_ENABLED;
    public final Property<Boolean> APP_CLEAR_CONFIG_AT_SHUTDOWN_ENABLED;
    public final Property<Boolean> APP_CFG_DYNAMIC_POLLING_ENABLED;

    //
    // ~ Sentry
    //
    public final Property<Boolean> SENTRY_ALERTS_FILTER_ENABLED;
    public final Property<List<Class<? extends Throwable>>> SENTRY_ALERTS_EX_CLASSES_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_EX_MSGS_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_EX_PATTERNS_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_LOG_MESSAGES_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_LOGGERS_TO_IGNORE;

    private final ApplicationConfig cfg;
}
