package com.turbospaces.cfg;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.ConfigurableCloudConnector;
import org.springframework.util.ResourceUtils;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.netflix.archaius.api.Property;
import com.netflix.archaius.api.config.PollingStrategy;
import com.netflix.archaius.api.config.SettableConfig;
import com.netflix.archaius.api.exceptions.ConfigException;
import com.netflix.archaius.config.DefaultCompositeConfig;
import com.netflix.archaius.config.DefaultSettableConfig;
import com.netflix.archaius.config.EmptyConfig;
import com.netflix.archaius.config.MapConfig;
import com.netflix.archaius.config.SystemConfig;
import com.netflix.archaius.config.polling.FixedPollingStrategy;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import com.typesafe.config.ConfigSyntax;
import com.typesafe.config.ConfigValue;

import reactor.core.Disposable;

@SuppressWarnings("unchecked")
public class ApplicationConfig extends DefaultCompositeConfig implements DynamicCompositeConfig, CloudOptions, PollingStrategy, Disposable {
    private final Logger logger = LoggerFactory.getLogger( ApplicationConfig.class );
    private final PollingStrategy pollingStrategy;
    private final DynamicPropertyFactory propertyFactory;

    public ApplicationConfig() throws ConfigException {
        //
        // ~ ignore some of the patterns
        //
        List<String> ignorePattern = new LinkedList<>();
        ignorePattern.add( ConfigurableCloudConnector.ENV_UPS_PREFIX );
        ignorePattern.add( ConfigurableCloudConnector.ENV_SPACE_NAME );
        ignorePattern.add( ConfigurableCloudConnector.ENV_HOSTNAME );

        // replace
        Map<String, String> oldgetenv = new HashMap<>( System.getenv() );
        Map<String, String> newgetenv = new HashMap<>();
        for ( Entry<String, String> it : oldgetenv.entrySet() ) {
            String oldProp = it.getKey();
            String newProp = oldProp.toLowerCase().replaceAll( "_", "." );

            boolean allowed = true;
            for ( String s : ignorePattern ) {
                if ( oldProp.startsWith( s ) ) {
                    allowed = false;
                    break;
                }
            }
            if ( allowed ) {
                logger.trace( "adding ENV prop {}={}", oldProp, newProp, it.getValue() );
                newgetenv.put( newProp, it.getValue() );
            }
        }

        //
        // ~ ordered hierarchy
        //
        addConfig( RUNTIME_CFG_NAME, new DefaultSettableConfig() ); // runtime
        addConfig( LOCAL_CFG_NAME, new DefaultSettableConfig() ); // local-dev-%username%
        addConfig( CLOUD_CFG_NAME, new DefaultSettableConfig() ); // CFG -> replace later
        addConfig( CMD_LINE_CFG_NAME, new DefaultSettableConfig() ); // CLI arguments
        addConfig( SYSTEM_ENV_CFG_NAME, new MapConfig( newgetenv ) ); // converted ENV variables
        addConfig( SYSTEM_PROPS_CFG_NAME, new SystemConfig() );
        addConfig( GIT_CFG_NAME, new DefaultSettableConfig() ); // GIT
        addConfig( LEGACY_CFG_NAME, EmptyConfig.INSTANCE );
        addConfig( DEFAULT_APP_CFG_NAME, new DefaultSettableConfig() ); // default classpath

        this.propertyFactory = DynamicPropertyFactory.from( this );

        long freq = getLong( ApplicationProperties.APP_TIMER_INTERVAL_KEY, 30L );
        this.pollingStrategy = new FixedPollingStrategy( freq, TimeUnit.SECONDS );
    }
    @Override
    public void dispose() {
        shutdown();
    }
    @Override
    public Future<?> execute(Runnable run) {
        return pollingStrategy.execute( run );
    }
    @Override
    public void shutdown() {
        pollingStrategy.shutdown();
    }
    @Override
    public DynamicPropertyFactory factory() {
        return propertyFactory;
    }
    @Override
    public List<String> getStringList(Property<String> prop) {
        List<String> l = new LinkedList<>();

        if ( containsKey( prop.getKey() ) ) {
            Object raw = getRawProperty( prop.getKey() );
            if ( raw instanceof String ) {
                l.addAll( getList( prop.getKey(), String.class ) );
            }
            else {
                l.addAll( (Collection<? extends String>) raw );
            }
        }
        else {
            String value = prop.get();
            if ( StringUtils.isNotEmpty( value ) ) {
                String[] split = value.split( getListDelimiter() );
                for ( String s : split ) {
                    l.add( s );
                }
            }
        }

        return ImmutableList.copyOf( Iterables.filter( l, new Predicate<String>() {
            @Override
            public boolean apply(final String input) {
                return StringUtils.isNotEmpty( input );
            }
        } ) );
    }
    @Override
    public List<Integer> getIntList(Property<Integer> prop) {
        List<Integer> l = new LinkedList<>();
        if ( containsKey( prop.getKey() ) ) {
            Object raw = getRawProperty( prop.getKey() );
            if ( raw instanceof String ) {
                l.addAll( getList( prop.getKey(), Integer.class ) );
            }
            else {
                l.addAll( (Collection<? extends Integer>) raw );
            }
        }
        else {
            l.add( prop.get() );
        }
        return l;
    }
    @Override
    public List<Long> getLongList(Property<Long> prop) {
        List<Long> l = new LinkedList<>();
        if ( containsKey( prop.getKey() ) ) {
            Object raw = getRawProperty( prop.getKey() );
            if ( raw instanceof String ) {
                l.addAll( getList( prop.getKey(), Long.class ) );
            }
            else {
                l.addAll( (Collection<? extends Long>) raw );
            }
        }
        else {
            l.add( prop.get() );
        }
        return l;
    }
    @Override
    public void clearLocalProperty(String key) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( LOCAL_CFG_NAME );
        c.clearProperty( key );
        logger.info( "clearing local prop: {}", key );
    }
    @Override
    public void setLocalProperty(String key, Object value) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( LOCAL_CFG_NAME );
        c.setProperty( key, value );
        logger.info( "setting local prop: {} -> {}", key, value );
    }
    @Override
    public void setLocalProperties(Properties props) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( LOCAL_CFG_NAME );
        c.setProperties( props );
        logger.info( "setting local props: {}", props );
    }
    @Override
    public void clearGitProperty(String key) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( GIT_CFG_NAME );
        c.clearProperty( key );
        logger.info( "clearing git prop: {}", key );
    }
    @Override
    public void setGitProperty(String key, Object value) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( GIT_CFG_NAME );
        c.setProperty( key, value );
        logger.info( "setting git prop: {} -> {}", key, value );
    }
    @Override
    public void setGitProperties(Properties props) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( GIT_CFG_NAME );
        c.setProperties( props );
        logger.info( "setting git props: {}", props );
    }
    @Override
    public void clearCmdLineParams(String key) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( CMD_LINE_CFG_NAME );
        c.clearProperty( key );
        logger.info( "clearing command line prop: {}", key );
    }
    @Override
    public void setCmdLineParams(String key, Object value) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( CMD_LINE_CFG_NAME );
        c.setProperty( key, value );
        logger.info( "setting command line prop: {} -> {}", key, value );
    }
    @Override
    public void setCmdLineParams(Properties props) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( CMD_LINE_CFG_NAME );
        c.setProperties( props );
        logger.info( "setting command line props: {}", props );
    }
    @Override
    public void clearDefaultProperty(String key) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( DEFAULT_APP_CFG_NAME );
        c.clearProperty( key );
        logger.info( "clearing default prop: {}", key );
    }
    @Override
    public void setDefaultProperty(String key, Object value) {
        SettableConfig c = (SettableConfig) getConfig( DEFAULT_APP_CFG_NAME );
        c.setProperty( key, value );
        logger.info( "setting default prop: {} -> {}", key, value );
    }
    @Override
    public void setDefaultProperties(Properties props) {
        DefaultSettableConfig c = (DefaultSettableConfig) getConfig( DEFAULT_APP_CFG_NAME );
        c.setProperties( props );
        logger.info( "setting default props: {}", props );
    }
    @Override
    public void clearAllDefaultProperties() throws ConfigException {
        replaceConfig( DEFAULT_APP_CFG_NAME, new DefaultSettableConfig() );
    }
    @Override
    public ApplicationConfig loadLocalDevProperties(String username) throws Exception {
        String userOverrideProps = String.format( ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-%s.properties", username );
        String userOverrideCfg = String.format( ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-%s.conf", username );

        URL propsURL = null;
        URL cfgURL = null;

        try {
            propsURL = ResourceUtils.getURL( userOverrideProps );
        }
        catch ( FileNotFoundException err ) {
            try {
                userOverrideProps = ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-template.properties";
                logger.debug( "no user specific props file found, will try to lookup = '{}'", userOverrideProps );
                propsURL = ResourceUtils.getURL( userOverrideProps );
            }
            catch ( FileNotFoundException io ) {

            }
        }

        try {
            cfgURL = ResourceUtils.getURL( userOverrideCfg );
        }
        catch ( FileNotFoundException err ) {
            try {
                userOverrideCfg = ResourceUtils.CLASSPATH_URL_PREFIX + "local-dev-template.conf";
                logger.debug( "no user specific cfg file found, attempting to lookup = '{}'", userOverrideCfg );
                cfgURL = ResourceUtils.getURL( userOverrideCfg );
            }
            catch ( FileNotFoundException io ) {

            }
        }

        if ( propsURL != null ) {
            try (InputStream io = propsURL.openStream()) {
                Properties tmp = new Properties();
                tmp.load( io );
                setLocalProperties( tmp );
            }
        }

        if ( cfgURL != null ) {
            File file = new File( cfgURL.toURI() );

            ConfigParseOptions parseOptions = ConfigParseOptions.defaults().setAllowMissing( false ).setSyntax( ConfigSyntax.CONF );
            Config typesafeConfig = ConfigFactory.parseFile( file, parseOptions ).resolve();

            for ( Map.Entry<String, ConfigValue> entry : typesafeConfig.entrySet() ) {
                String key = entry.getKey();
                Object value = entry.getValue().unwrapped();
                setLocalProperty( key, value );
            }
        }

        return this;
    }
    @Override
    public ApplicationConfig loadDefaultPropsFromResource(URL resource) throws IOException {
        try (InputStream io = resource.openStream()) {
            Properties props = new Properties();
            props.load( io );
            setDefaultProperties( props );
        }
        return this;
    }
    @Override
    public ApplicationConfig loadLocalDevProperties() throws Exception {
        String username = System.getProperty( "user.name" );
        loadLocalDevProperties( username );
        return this;
    }
    public static ApplicationConfig create() throws ConfigException {
        return new ApplicationConfig();
    }
    public static ApplicationConfig create(String appId) throws ConfigException {
        ApplicationConfig cfg = create();
        cfg.setDefaultProperty( CloudOptions.CLOUD_APP_ID, appId );
        return cfg;
    }
}
