package com.turbospaces.common;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URL;
import java.net.UnknownHostException;
import java.nio.channels.DatagramChannel;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.netflix.archaius.api.Property;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.cfg.CloudOptions;

public abstract class PlatformUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(PlatformUtil.class);

    public static final Map<String, String> ISO2_CODES, ISO3_CODES;

    static {
        String[] countries = Locale.getISOCountries();
        Map<String, String> m = new HashMap<>(countries.length);
        for (String country : countries) {
            Locale locale = new Locale(StringUtils.EMPTY, country);
            m.put(country, locale.getISO3Country().toUpperCase());
        }

        ISO2_CODES = ImmutableMap.copyOf(m);
    }

    static {
        String[] countries = Locale.getISOCountries();
        Map<String, String> m = new HashMap<>(countries.length);
        for (String country : countries) {
            Locale locale = new Locale(StringUtils.EMPTY, country);
            m.put(locale.getISO3Country().toUpperCase(), locale.getCountry().toUpperCase());
        }

        ISO3_CODES = ImmutableMap.copyOf(m);
    }

    public static InetSocketAddress address(ApplicationConfig cfg) {
        String ip = cfg.getString(CloudOptions.CLOUD_APP_HOST);
        int port = cfg.getInteger(CloudOptions.CLOUD_APP_PORT);
        return new InetSocketAddress(ip, port);
    }
    public static InetSocketAddress address(Properties props) {
        String ip = props.get(CloudOptions.CLOUD_APP_HOST).toString();
        String port = props.get(CloudOptions.CLOUD_APP_PORT).toString();
        return new InetSocketAddress(ip, Integer.parseInt(port));
    }
    public static List<String> readStaticConstants(Class<?> type) throws IllegalAccessException {
        Preconditions.checkArgument(type.isInterface());

        List<String> l = Lists.newLinkedList();
        Field[] fields = FieldUtils.getAllFields(type);

        for (Field f : fields) {
            if (Modifier.isPublic(f.getModifiers()) && BooleanUtils.isTrue(Modifier.isStatic(f.getModifiers()))) {
                String key = (String) FieldUtils.readStaticField(f);
                l.add(key);
            }
        }

        return l;
    }
    public static Optional<String> jarVersion(Class<?> clazz) {
        try (InputStream stream = clazz.getClassLoader().getResourceAsStream("/META-INF/MANIFEST.MF")) {
            Manifest manifest = new Manifest(stream);
            Attributes attributes = manifest.getMainAttributes();
            String version = attributes.getValue("Implementation-Version");
            if (version != null) {
                return Optional.of(version);
            }
        } catch (Exception err) {
            LOGGER.trace(err.getMessage(), err);
        }

        String version = clazz.getPackage().getImplementationVersion();
        if (version != null) {
            return Optional.of(version);
        }

        try {
            URL location = clazz.getProtectionDomain().getCodeSource().getLocation();
            if (location.getFile() != null) {
                return jarVersion(new File(location.getFile()));
            }
        } catch (Exception err) {
            LOGGER.trace(err.getMessage(), err);
        }

        return Optional.empty();
    }
    public static Optional<String> jarVersion(File f) throws IOException {
        try (java.util.jar.JarFile jar = new java.util.jar.JarFile(f)) {
            java.util.jar.Manifest manifest = jar.getManifest();
            java.util.jar.Attributes attributes = manifest.getMainAttributes();

            String impVersion = attributes.getValue("Implementation-Version");
            if (impVersion != null) {
                return Optional.of(impVersion);
            }
        }
        return Optional.empty();
    }
    public static String version(Property<String> prop) {
        String appName = prop.get();
        if (StringUtils.isNotEmpty(appName)) {
            String[] split = appName.split(":");
            if (split.length == 3) { // maven coordinates
                String groupId = split[0];
                String artifactId = split[1];
                String version = split[2];
                if (StringUtils.isNotEmpty(groupId) && StringUtils.isNotEmpty(artifactId)) {
                    return version;
                }
            }
        }

        Optional<String> opt = PlatformUtil.jarVersion(PlatformUtil.class);
        if (opt.isPresent()) {
            return opt.get();
        }

        return "latest".toUpperCase().intern();
    }
    public static List<Runnable> shutdownExecutor(ExecutorService executor, Duration timeout) {
        List<Runnable> runnables = Collections.emptyList();
        executor.shutdown(); // Disable new tasks from being submitted.
        try {
            // Wait a while for existing tasks to terminate.
            boolean allCompleted = executor.awaitTermination(timeout.toSeconds(), TimeUnit.SECONDS);
            if (BooleanUtils.isFalse(allCompleted)) {
                runnables = executor.shutdownNow(); // Cancel currently executing tasks.
                LOGGER.debug("listing {} tasks that never commenced execution ...", runnables.size());
                for (Runnable runnable : runnables) {
                    LOGGER.debug(runnable.toString());
                }
            }
        } catch (InterruptedException ie) {
            LOGGER.debug(ie.getMessage(), ie);
            executor.shutdownNow();
            Thread.currentThread().interrupt();
        }
        return runnables;
    }
    public static String detectIp() {
        try (DatagramChannel channel = DatagramChannel.open().connect(new InetSocketAddress("8.8.8.8", 1))) {
            return ((InetSocketAddress) channel.getLocalAddress()).getHostString();
        } catch (Exception err) {
            LOGGER.debug(err.getMessage(), err);
            try {
                return InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException e) {
                throw new RuntimeException(e);
            }
        }
    }
    public static String fetchExternalIp(ApplicationProperties props) {
        try {
            URI uri = new URI("http://checkip.amazonaws.com");
            HttpURLConnection con = (HttpURLConnection) uri.toURL().openConnection();
            con.setRequestMethod("GET");
            con.setConnectTimeout((int) props.TCP_CONNECTION_TIMEOUT.get().toMillis());
            con.setReadTimeout((int) props.TCP_SOCKET_TIMEOUT.get().toMillis());

            StringBuilder sb = new StringBuilder();
            try (InputStream io = con.getInputStream()) {
                try (BufferedReader br = new BufferedReader(new InputStreamReader(io))) {
                    String output;
                    while ( (output = br.readLine()) != null ) {
                        sb.append(output);
                    }
                }
            }

            return sb.toString();
        } catch (Exception err) {
            LOGGER.debug(err.getMessage(), err);
            try {
                return InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException e) {
                throw new RuntimeException(e);
            }
        }
    }
    @SuppressWarnings("deprecation")
    public static int findAvailableTcpPort() {
        return org.springframework.util.SocketUtils.findAvailableTcpPort();
    }
    public static LocalDate toLocalDate(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }
    public static LocalDateTime toLocalDateTime(Date date) {
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
    }
    public static Date toDate(LocalDateTime date) {
        return Date.from(date.atZone(ZoneId.systemDefault()).toInstant());
    }
    public static LocalDate toLocalUTCDate() {
        return toLocalUTCDate(new Date());
    }
    public static LocalDate toLocalUTCDate(Date date) {
        return date.toInstant().atOffset(ZoneOffset.UTC).toLocalDate();
    }
    public static ToStringBuilder noClassNameToString(Object obj) {
        return new ToStringBuilder(obj, ToStringStyle.NO_CLASS_NAME_STYLE);
    }
    public static <T> Stream<T> takeWhile(Stream<T> stream, Predicate<T> predicate) {
        return StreamSupport.stream(new PredicateSpliterator<>(stream.spliterator(), predicate), false);
    }

    public static class PredicateSpliterator<T> extends Spliterators.AbstractSpliterator<T> {
        private final Spliterator<T> spliterator;
        private final Predicate<T> predicate;
        private boolean active = true;

        public PredicateSpliterator(Spliterator<T> spliterator, Predicate<T> predicate) {
            super(spliterator.estimateSize(), IMMUTABLE);
            this.spliterator = spliterator;
            this.predicate = predicate;
        }
        @Override
        public boolean tryAdvance(Consumer<? super T> consumer) {
            if (active) {
                boolean hasNext = spliterator.tryAdvance(new Consumer<T>() {
                    @Override
                    public void accept(T t) {
                        if (predicate.test(t)) {
                            consumer.accept(t);
                        } else {
                            active = false;
                        }
                    }
                });
                return hasNext && active;
            }
            return false;
        }
    }

    private PlatformUtil() {}
}
