package com.turbospaces.common.logging;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import java.util.Arrays;
import java.util.stream.Stream;

class MaskObfuscatorTest {
    public static Stream<Arguments> testObfuscateJsonData() {
        return Stream.of(
                Arguments.of(null, null),
                Arguments.of("", ""),
                Arguments.of("{}", "{}"),
                Arguments.of("{\"address\":\"***\"}", "{\"address\":\"foo\"}"),
                Arguments.of(
                        "{\n" +
                                "    \"user_id\":\"87656\",\n" +
                                "    \"address\" : \"************************\",\n" +
                                "    \"city\":\"Chicago\"\n" +
                                " }",
                        "{\n" +
                                "    \"user_id\":\"87656\",\n" +
                                "    \"address\" : \"hotel California, ap. 42\",\n" +
                                "    \"city\":\"Chicago\"\n" +
                                " }"
                ),
                Arguments.of("{\"password\":\"*****\",\"address\":\"************************\"}", "{\"password\":\"87656\",\"address\":\"hotel California, ap. 42\"}")
        );
    }

    @ParameterizedTest
    @MethodSource("testObfuscateJsonData")
    void testObfuscateJson(String expected, String msg) {
        MaskObfuscator jsonObfuscator = new MaskObfuscator(Arrays.asList(
                Obfuscators.jsonStrFieldMaskPattern("address"),
                Obfuscators.jsonStrFieldMaskPattern("password")
        ));
        Assertions.assertEquals(expected, jsonObfuscator.obfuscate(msg));
    }

    public static Stream<Arguments> testObfuscateQueryData() {
        return Stream.of(
                Arguments.of("", ""),
                Arguments.of("{}", "{}"),
                Arguments.of("https://skrill.com/app/pay.pl?password=***", "https://skrill.com/app/pay.pl?password=foo"),
                Arguments.of(
                        "prepare req: GET https://skrill.com/app/pay.pl" +
                                "?action=prepare&email=paysafe%40ysi-group.com&password=********************************&amount=150.00" +
                                "&currency=***&transaction_id=3d8224a1 HTTP/1.1",
                        "prepare req: GET https://skrill.com/app/pay.pl" +
                                "?action=prepare&email=paysafe%40ysi-group.com&password=1acbf769dbd19ee3030199817d1f8966&amount=150.00" +
                                "&currency=USD&transaction_id=3d8224a1 HTTP/1.1")
        );
    }

    @ParameterizedTest
    @MethodSource("testObfuscateQueryData")
    void testObfuscateQuery(String expected, String msg) {
        MaskObfuscator queryObfuscator = new MaskObfuscator(Arrays.asList(
                Obfuscators.queryParamMaskPattern("password"),
                Obfuscators.queryParamMaskPattern("currency")
        ));
        Assertions.assertEquals(expected, queryObfuscator.obfuscate(msg));
    }
}