package com.turbospaces.boot;

import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.common.ThrowableAction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

public abstract class AbstractExecutionPlatform implements ExecutionPlatform {
    protected final Logger logger = LoggerFactory.getLogger(getClass());

    public AbstractExecutionPlatform(ApplicationProperties props) {
        this.props = props;
    }

    protected final ApplicationProperties props;


    @Override
    public void shutdown() {
        executor().shutdown();
    }

    @Override
    public List<Runnable> shutdownNow() {
        return executor().shutdownNow();
    }

    @Override
    public boolean isShutdown() {
        return executor().isShutdown();
    }

    @Override
    public boolean isTerminated() {
        return executor().isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return executor().awaitTermination(timeout, unit);
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
        return executor().submit(task);
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
        return executor().submit(task, result);
    }

    @Override
    public Future<?> submit(Runnable task) {
        return executor().submit(task);
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
        return executor().invokeAll(tasks);
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException {
        return executor().invokeAll(tasks, timeout, unit);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
        return executor().invokeAny(tasks);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        return executor().invokeAny(tasks, timeout, unit);
    }

    @Override
    public void execute(Runnable command) {
        Map<String, String> mdc = MDC.getCopyOfContextMap(); // ~ capture MDC
        if (mdc == null) {
            executor().execute(command);
        } else {
            executor().execute(new Runnable() {
                @Override
                public void run() {
                    MDC.setContextMap(mdc); // ~ set MDC stuff back if necessary
                    try {
                        command.run();
                    } catch (Throwable err) {
                        logger.error(err.getMessage(), err);
                    } finally {
                        MDC.clear(); // ~ finally clear it if necessary
                    }
                }
            });
        }
    }

    @Override
    public void work(ThrowableAction action) {
        execute(new Runnable() {
            @Override
            public void run() {
                try {
                    action.apply();
                } catch (Exception err) {
                    logger.error(err.getMessage(), err);
                }
            }
        });
    }

    @Override
    public void dispose() {
        int timeout = props.APP_PLATFORM_GRACEFUL_TIMEOUT.get();
        PlatformUtil.shutdownExecutor(executor(), timeout);
        logger.debug("stopped platform-worker pool ...");
    }
}
