package com.turbospaces.boot;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.net.URL;
import java.security.KeyStore;
import java.security.Security;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Supplier;

import com.turbospaces.common.logging.Obfuscators;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.mutable.MutableObject;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.DestructionAwareBeanPostProcessor;
import org.springframework.boot.BootstrapContextClosedEvent;
import org.springframework.boot.BootstrapRegistry;
import org.springframework.boot.BootstrapRegistryInitializer;
import org.springframework.boot.DefaultApplicationArguments;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.context.properties.source.ConfigurationPropertySources;
import org.springframework.boot.logging.LoggingSystem;
import org.springframework.cloud.Cloud;
import org.springframework.cloud.ConfigurableCloudConnector;
import org.springframework.cloud.ConfigurableCloudFactory;
import org.springframework.cloud.app.ApplicationInstanceInfo;
import org.springframework.cloud.service.UriBasedServiceInfo;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.event.SmartApplicationListener;
import org.springframework.core.ResolvableType;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.core.env.ConfigurablePropertyResolver;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertySource;
import org.springframework.util.ResourceUtils;

import com.codahale.metrics.ConsoleReporter;
import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Slf4jReporter;
import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheck.Result;
import com.codahale.metrics.health.HealthCheckFilter;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.codahale.metrics.health.HealthCheckRegistryListener;
import com.codahale.metrics.jmx.JmxReporter;
import com.codahale.metrics.jvm.ThreadDump;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.Uninterruptibles;
import com.netflix.archaius.api.Config;
import com.netflix.archaius.config.DefaultConfigListener;
import com.netflix.archaius.config.PollingDynamicConfig;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.cfg.DynamicCompositeConfig;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.logging.AlertLoggingFilter;
import com.turbospaces.logging.Logback;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.UPSs;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import io.jaegertracing.Configuration;
import io.jaegertracing.internal.Constants;
import io.jaegertracing.internal.JaegerTracer;
import io.micrometer.core.instrument.Clock;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.binder.MeterBinder;
import io.micrometer.core.instrument.binder.jvm.ClassLoaderMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmCompilationMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmGcMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmHeapPressureMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmInfoMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmMemoryMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmThreadMetrics;
import io.micrometer.core.instrument.binder.logging.LogbackMetrics;
import io.micrometer.core.instrument.binder.system.FileDescriptorMetrics;
import io.micrometer.core.instrument.binder.system.ProcessorMetrics;
import io.micrometer.core.instrument.composite.CompositeMeterRegistry;
import io.micrometer.core.instrument.dropwizard.DropwizardConfig;
import io.micrometer.core.instrument.dropwizard.DropwizardMeterRegistry;
import io.micrometer.core.instrument.util.HierarchicalNameMapper;
import io.micrometer.elastic.ElasticConfig;
import io.micrometer.elastic.ElasticMeterRegistry;
import io.micrometer.influx.InfluxConfig;
import io.micrometer.influx.InfluxMeterRegistry;
import io.opentracing.Tracer;
import io.sentry.SentryClient;
import io.sentry.SentryClientFactory;
import io.sentry.connection.EventSendCallback;
import io.sentry.event.Event;
import io.sentry.event.helper.ShouldSendEventCallback;

public abstract class AbstractBootstrap<PROPS extends ApplicationProperties> extends SpringApplication implements Bootstrap, ShouldSendEventCallback {
    static {
        Security.addProvider( new BouncyCastleProvider() );
    }

    private final Logger logger = LoggerFactory.getLogger( getClass() );
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final MetricRegistry metricRegistry = new MetricRegistry();
    private final CompositeMeterRegistry meterRegistry = Metrics.globalRegistry;
    private final List<MeterBinder> binders = Lists.newLinkedList();

    private final Set<Channel> channels = new LinkedHashSet<>();

    private final JmxReporter jmxReporter;
    private final ConsoleReporter consoleReporter;
    private final HealthCheckRegistry healthCheckRegistry;
    private final Thread shutdownHook;
    private final String release;

    private final Cloud cloud;
    private final PROPS props;
    private final AbstractEnvironment env;

    private ConfigurableApplicationContext applicationContext;

    private final FixedSizePlatform platform;
    private final SentryClient sentry;
    private final JaegerTracer tracer;

    private KeyStore keyStore;

    protected AbstractBootstrap(Class<?> mainClass, PROPS props) throws Exception {
        super( mainClass );

        this.keyStore = KeyStore.getInstance( KeyStore.getDefaultType() );
        this.props = Objects.requireNonNull( props );

        env = new AbstractEnvironment() {
            @Override
            protected void customizePropertySources(MutablePropertySources sources) {
                sources.addFirst( new PropertySource<Object>( "archaius" ) {
                    @Override
                    public Object getProperty(String key) {
                        return props.cfg().getRawProperty( key );
                    }
                } );
            }
            @Override
            protected ConfigurablePropertyResolver createPropertyResolver(MutablePropertySources sources) {
                return ConfigurationPropertySources.createPropertyResolver( sources );
            }
        };

        setWebApplicationType( WebApplicationType.NONE );
        setLogStartupInfo( false );
        setEnvironment( env );
        setRegisterShutdownHook( false );

        ConfigurableCloudConnector connector = new ConfigurableCloudConnector( props, keyStore );
        ApplicationInstanceInfo info = connector.getApplicationInstanceInfo();
        Map<String, Object> cloudProps = info.getProperties();

        String space = cloudProps.get( CloudOptions.CLOUD_APP_SPACE_NAME ).toString();
        String host = cloudProps.get( CloudOptions.CLOUD_APP_HOST ).toString();
        String slot = cloudProps.get( CloudOptions.CLOUD_APP_INSTANCE_INDEX ).toString();
        String service = info.getAppId();

        this.release = PlatformUtil.version( props.CLOUD_APP_NAME );

        // ~ sentry
        Optional<UriBasedServiceInfo> sentryOpt = UPSs.findServiceInfoByName( connector, UPSs.SENTRY );
        if ( sentryOpt.isPresent() ) {
            UriBasedServiceInfo serviceInfo = sentryOpt.get();
            sentry = SentryClientFactory.sentryClient( serviceInfo.getUri() );
        }
        else {
            sentry = SentryClientFactory.sentryClient();
        }
        sentry.setEnvironment( space );
        sentry.setRelease( release );
        sentry.setServerName( service + "/" + slot );
        sentry.addShouldSendEventCallback( this );
        sentry.addEventSendCallback( new EventSendCallback() {
            @Override
            public void onSuccess(Event event) {
                Counter counter = metricRegistry.counter( MetricRegistry.name( "sentry", "success" ) );
                counter.inc();
            }
            @Override
            public void onFailure(Event event, Exception exception) {
                Counter counter = metricRegistry.counter( MetricRegistry.name( "sentry", "failure" ) );
                counter.inc();
            }
        } );

        // ~ prepare logger context
        LoggerContext loggerFactory = (LoggerContext) LoggerFactory.getILoggerFactory();
        loggerFactory.setPackagingDataEnabled( isDevMode() );

        // ~ INFO
        String[] infoByDefault = new String[] { "org.eclipse.jgit" };
        for ( String it : infoByDefault ) {
            ch.qos.logback.classic.Logger log = loggerFactory.getLogger( it );
            log.setLevel( Level.INFO );
        }

        // ~ cloud
        MutableObject<URL> logFile = new MutableObject<>();
        ConfigurableCloudFactory cloudFactory = new ConfigurableCloudFactory( connector, this, logFile );
        cloud = cloudFactory.getCloud();
        configureLogging( logFile );

        // ~ tracer
        ImmutableMap.Builder<String, String> tracingTags = ImmutableMap.builder();
        tracingTags.put( Constants.TRACER_IP_TAG_KEY, PlatformUtil.detectIp() );
        tracingTags.put( Constants.TRACER_HOSTNAME_TAG_KEY, host );
        tracingTags.put( CloudOptions.CLOUD_APP_INSTANCE_INDEX, slot );

        Configuration fromEnv = Configuration.fromEnv( service ).withTracerTags( tracingTags.build() );
        tracer = fromEnv.getTracer();

        ImmutableList.Builder<Tag> meterTags = ImmutableList.builder();
        meterTags.add( Tag.of( "env", space ) );
        meterTags.add( Tag.of( "release", release ) );
        meterTags.add( Tag.of( "service", service ) );
        meterRegistry.config().commonTags( meterTags.build() );

        meterRegistry.add( new DropwizardMeterRegistry( new DropwizardConfig() {
            @Override
            public String prefix() {
                return "boot";
            }
            @Override
            public String get(String key) {
                return props.cfg().getString( key, null );
            }
        }, metricRegistry, HierarchicalNameMapper.DEFAULT, Clock.SYSTEM ) {
            @Override
            protected Double nullGaugeValue() {
                return Double.NaN;
            }
        } );

        //
        // ~ out of the box reasonable metrics
        //
        binders.add( new FileDescriptorMetrics() );
        binders.add( new ClassLoaderMetrics() );
        binders.add( new JvmGcMetrics() );
        binders.add( new ProcessorMetrics() );
        binders.add( new JvmThreadMetrics() );
        binders.add( new JvmHeapPressureMetrics() );
        binders.add( new JvmCompilationMetrics() );
        binders.add( new JvmInfoMetrics() );
        binders.add( new JvmMemoryMetrics() );
        binders.add( new LogbackMetrics() );

        if ( props.APP_METRICS_DRY_RUN.get() ) {}
        else {
            Optional<PlainServiceInfo> optelk = UPSs.findServiceInfoByName( this, UPSs.ELASTIC_SEARCH );
            Optional<PlainServiceInfo> optinf = UPSs.findServiceInfoByName( this, UPSs.INFLUX );

            if ( optelk.isPresent() ) {
                PlainServiceInfo si = optelk.get();

                if ( props.APP_METRICS_ELK_REPORTER_ENABLED.get() ) {
                    meterRegistry.add( new ElasticMeterRegistry( new ElasticConfig() {
                        @Override
                        public int batchSize() {
                            return props.APP_METRICS_BULK_SIZE.get();
                        }
                        @Override
                        public String host() {
                            return String.format( "%s://%s:%d", si.getScheme(), si.getHost(), si.getPort() );
                        }
                        @Override
                        public String userName() {
                            return si.getUserName();
                        }
                        @Override
                        public String password() {
                            return si.getPassword();
                        }
                        @Override
                        public Duration connectTimeout() {
                            return Duration.ofSeconds( props.TCP_CONNECTION_TIMEOUT.get() );
                        }
                        @Override
                        public Duration readTimeout() {
                            return Duration.ofSeconds( props.TCP_SOCKET_TIMEOUT.get() );
                        }
                        @Override
                        public String get(String k) {
                            return props.cfg().getString( k, null );
                        }
                    }, Clock.SYSTEM ) );
                }
            }

            if ( optinf.isPresent() ) {
                PlainServiceInfo si = optinf.get();

                if ( props.APP_METRICS_INFLUX_REPORTER_ENABLED.get() ) {
                    meterRegistry.add( new InfluxMeterRegistry( new InfluxConfig() {
                        @Override
                        public int batchSize() {
                            return props.APP_METRICS_BULK_SIZE.get();
                        }
                        @Override
                        public String uri() {
                            return String.format( "%s://%s:%d", si.getScheme(), si.getHost(), si.getPort() );
                        }
                        @Override
                        public String userName() {
                            return si.getUserName();
                        }
                        @Override
                        public String password() {
                            return si.getPassword();
                        }
                        @Override
                        public String db() {
                            return si.getPath();
                        }
                        @Override
                        public Duration connectTimeout() {
                            return Duration.ofSeconds( props.TCP_CONNECTION_TIMEOUT.get() );
                        }
                        @Override
                        public Duration readTimeout() {
                            return Duration.ofSeconds( props.TCP_SOCKET_TIMEOUT.get() );
                        }
                        @Override
                        public String get(String k) {
                            return props.cfg().getString( k, null );
                        }
                    }, Clock.SYSTEM ) );
                }
            }
        }

        for ( MeterBinder binder : binders ) {
            binder.bindTo( meterRegistry() );
        }

        // ~ thread pool
        platform = new FixedSizePlatform( props, meterRegistry );

        // ~ JMX/console reporters
        jmxReporter = JmxReporter.forRegistry( metricRegistry ).inDomain( props.APP_JMX_DOMAIN.get() ).build();
        consoleReporter = ConsoleReporter.forRegistry( metricRegistry ).scheduleOn( platform ).build();

        // ~ shutdown hook
        shutdownHook = new Thread( new Runnable() {
            @Override
            public void run() {
                logger.info( "running shutdown hook now ..." );
                try {
                    shutdown();
                }
                catch ( Throwable err ) {
                    logger.error( err.getMessage(), err );
                }
            }
        } );

        // ~ health check registry
        healthCheckRegistry = new HealthCheckRegistry();
        healthCheckRegistry.addListener( new HealthCheckRegistryListener() {
            @Override
            public void onHealthCheckAdded(String name, HealthCheck healthCheck) {
                if ( healthCheck instanceof BootstrapAware ) {
                    try {
                        ( (BootstrapAware) healthCheck ).setBootstrap( AbstractBootstrap.this );
                    }
                    catch ( Exception err ) {
                        ExceptionUtils.wrapAndThrow( err );
                    }
                }
            }
            @Override
            public void onHealthCheckRemoved(String name, HealthCheck healthCheck) {
                if ( healthCheck instanceof AbstractHealtchCheck ) {
                    AbstractHealtchCheck preDestoy = (AbstractHealtchCheck) healthCheck;
                    try {
                        preDestoy.destroy();
                    }
                    catch ( Exception err ) {
                        logger.error( err.getMessage(), err );
                    }
                }
            }
        } );
    }
    @Override
    public ConfigurableApplicationContext run(String... args) {
        boolean isHealthy = true;
        Set<String> unhealthy = new HashSet<>();

        Lock rwLock = lock.writeLock();
        rwLock.lock();
        try {
            if ( props.APP_WAIT_FOR_HEALTHCHECKS_ENABLED.get() ) {
                logger.debug( "about to run health-checks now ..." );

                int it = 0;
                long now = System.currentTimeMillis();
                long timeout = TimeUnit.SECONDS.toMillis( props.APP_WAIT_FOR_HEALTHCHECKS_TIMEOUT.get() );
                isHealthy = false;

                while ( ( System.currentTimeMillis() - now ) <= timeout ) {
                    boolean tmp = true;
                    it++;
                    unhealthy.clear();

                    // ~ we want to run regular health-checks or just those which are meant for boot only
                    for ( Entry<String, Result> entry : healthCheckRegistry().runHealthChecks( new HealthCheckFilter() {
                        @Override
                        public boolean matches(String name, HealthCheck healthCheck) {
                            if ( healthCheck instanceof AbstractHealtchCheck ) {
                                AbstractHealtchCheck check = (AbstractHealtchCheck) healthCheck;
                                return check.isBootstrapOnly();
                            }
                            return true;
                        }
                    } ).entrySet() ) {
                        logger.debug( "iteration({}) ::: healthcheck({}) - isHealthy({})", it, entry.getKey(), entry.getValue().isHealthy() );

                        tmp &= entry.getValue().isHealthy();
                        if ( entry.getValue().isHealthy() ) {}
                        else {
                            unhealthy.add( entry.getKey() );
                        }
                    }
                    if ( tmp ) {
                        isHealthy = true;
                        break;
                    }
                    int waitSec = props.APP_WAIT_FOR_HEALTHCHECKS_INTERVAL.get();
                    logger.debug( "about to wait {} sec before next health_check attempt ...", waitSec );
                    Uninterruptibles.sleepUninterruptibly( waitSec, TimeUnit.SECONDS );
                }
            }

            if ( isHealthy ) {
                long now = System.currentTimeMillis();

                logger.debug( "about to perform actual application start ..." );
                doStart( args );
                logger.info( "application started in={} sec ...", TimeUnit.MILLISECONDS.toSeconds( System.currentTimeMillis() - now ) );

                // register shutdown hook
                if ( props.APP_SHUTDOWN_HOOK_ENABLED.get() ) {
                    logger.info( "shutdown hook has been registered ..." );
                    Runtime.getRuntime().addShutdownHook( shutdownHook );
                }

                return applicationContext;
            }

            logger.info( "app will not start due to failed health checks ..." );
            removeAllHealthChecks();
            if ( platform != null ) {
                platform.dispose();
            }

            throw new Throwable( "unhealthy" + " = " + unhealthy.toString() );
        }
        catch ( Throwable err ) {
            // ~ log only if healthy
            if ( isHealthy ) {
                logger.error( err.getMessage(), err );
            }

            //
            // ~ remove hook
            //
            boolean removed = Runtime.getRuntime().removeShutdownHook( shutdownHook );
            if ( removed ) {
                logger.info( "removed shutdown hook ..." );
            }

            //
            // ~ try to perform clean shutdown
            //
            if ( isHealthy ) {
                logger.error( "application failed to start, stopping lifecycle thread ..." );
                try {
                    shutdown();
                }
                catch ( Throwable t ) {
                    logger.warn( t.getMessage(), t );
                }
            }

            //
            // ~ we need to dump all threads for better troubleshooting
            //
            logger.info( "dumping all threads now before termination ... " );
            ThreadMXBean threadMXBean = ManagementFactory.getThreadMXBean();
            ThreadDump dump = new ThreadDump( threadMXBean );
            dump.dump( System.err );

            // ~ post alert if application fails to start
            if ( isDevMode() ) {}
            else {
                try (Slf4jReporter reporter = Slf4jReporter.forRegistry( metricRegistry ).scheduleOn( platform ).build()) {
                    reporter.report();
                }
            }

            //
            // ~ re-throw error which will cause JVM exit with non-zero code
            //
            Throwables.throwIfUnchecked( err );
            throw new RuntimeException( err ); // docker(restart_policy=on-failure)
        }
        finally {
            rwLock.unlock();
        }
    }
    @Override
    protected void postProcessApplicationContext(ConfigurableApplicationContext context) {
        super.postProcessApplicationContext( context );

        for ( Channel channel : channels ) {
            if ( channel instanceof SmartApplicationListener ) {
                context.addApplicationListener( (SmartApplicationListener) channel );
            }
            if ( channel instanceof BeanFactoryPostProcessor ) {
                context.addBeanFactoryPostProcessor( (BeanFactoryPostProcessor) channel );
            }
        }
    }
    @Override
    public void shutdown() throws Exception {
        Lock rwLock = lock.writeLock();
        rwLock.lock();
        try {
            doStop();
        }
        finally {
            rwLock.unlock();
        }
    }
    @Override
    public void exit(int code) {
        Lock rwLock = lock.writeLock();
        rwLock.lock();
        try {
            if ( shutdownHook != null ) {
                Runtime.getRuntime().removeShutdownHook( shutdownHook );
            }
            System.exit( code );
        }
        finally {
            rwLock.unlock();
        }
    }
    @Override
    public PROPS props() {
        return props;
    }
    @Override
    public ApplicationConfig cfg() {
        return props.cfg();
    }
    @Override
    public FixedSizePlatform platform() {
        return platform;
    }
    @Override
    public MetricRegistry metricRegistry() {
        return metricRegistry;
    }
    @Override
    public MeterRegistry meterRegistry() {
        return meterRegistry;
    }
    @Override
    public HealthCheckRegistry healthCheckRegistry() {
        return healthCheckRegistry;
    }
    @Override
    public Cloud cloud() {
        return cloud;
    }
    @Override
    public ConfigurableApplicationContext context() {
        return applicationContext;
    }
    @Override
    public KeyStore keyStore() {
        return keyStore;
    }
    @Override
    public String release() {
        return release;
    }
    @Override
    public boolean isDevMode() {
        return props.APP_DEV_MODE.get();
    }
    @Override
    public int port() {
        return props.CLOUD_APP_PORT.get();
    }
    @Override
    public int secondaryPort() {
        return props.CLOUD_APP_SECONDARY_PORT.get();
    }
    @Override
    public int tertiaryPort() {
        return props.CLOUD_APP_TERTIARY_PORT.get();
    }
    @Override
    public String spaceName() {
        return props.CLOUD_APP_SPACE_NAME.get();
    }
    @Override
    public String appId() {
        return props.CLOUD_APP_ID.get();
    }
    @Override
    public void addBootstrapRegistryInitializer(BootstrapRegistryInitializer initializer) {
        if ( initializer instanceof BootstrapAware ) {
            ( (BootstrapAware) initializer ).setBootstrap( this );
        }
        super.addBootstrapRegistryInitializer( initializer );
    }
    @Override
    public boolean addChannel(Channel acceptor) {
        if ( acceptor instanceof BootstrapAware ) {
            ( (BootstrapAware) acceptor ).setBootstrap( this );
        }
        return channels.add( acceptor );
    }
    @Override
    public boolean isHealthy() {
        SortedMap<String, HealthCheck.Result> results = healthCheckRegistry.runHealthChecks();
        return results.isEmpty();
    }
    @Override
    public void registerHealthCheck(String name, HealthCheck check) {
        healthCheckRegistry().register( name, check );
    }
    @Override
    public SentryClient sentry() {
        return sentry;
    }
    @Override
    public Tracer tracer() {
        return tracer;
    }
    @Override
    public boolean shouldSend(Event event) {
        return props.APP_SENTRY_ENABLED.get();
    }
    @Override
    public Object getBean(String name) throws BeansException {
        return context().getBean( name );
    }
    @Override
    public <T> T getBean(String name, Class<T> requiredType) throws BeansException {
        return context().getBean( name, requiredType );
    }
    @Override
    public Object getBean(String name, Object... args) throws BeansException {
        return context().getBean( name, args );
    }
    @Override
    public <T> T getBean(Class<T> requiredType) throws BeansException {
        return context().getBean( requiredType );
    }
    @Override
    public <T> T getBean(Class<T> requiredType, Object... args) throws BeansException {
        return context().getBean( requiredType, args );
    }
    @Override
    public <T> ObjectProvider<T> getBeanProvider(Class<T> requiredType) {
        return context().getBeanProvider( requiredType );
    }
    @Override
    public <T> ObjectProvider<T> getBeanProvider(ResolvableType requiredType) {
        return context().getBeanProvider( requiredType );
    }
    @Override
    public boolean containsBean(String name) {
        return context().containsBean( name );
    }
    @Override
    public boolean isSingleton(String name) throws NoSuchBeanDefinitionException {
        return context().isSingleton( name );
    }
    @Override
    public boolean isPrototype(String name) throws NoSuchBeanDefinitionException {
        return context().isPrototype( name );
    }
    @Override
    public boolean isTypeMatch(String name, ResolvableType typeToMatch) throws NoSuchBeanDefinitionException {
        return context().isTypeMatch( name, typeToMatch );
    }
    @Override
    public boolean isTypeMatch(String name, Class<?> typeToMatch) throws NoSuchBeanDefinitionException {
        return context().isTypeMatch( name, typeToMatch );
    }
    @Override
    public Class<?> getType(String name) throws NoSuchBeanDefinitionException {
        return context().getType( name );
    }
    @Override
    public Class<?> getType(String name, boolean allowFactoryBeanInit) throws NoSuchBeanDefinitionException {
        return context().getType( name, allowFactoryBeanInit );
    }
    @Override
    public String[] getAliases(String name) {
        return context().getAliases( name );
    }
    protected void doStart(String... args) throws Throwable {
        addBootstrapRegistryInitializer( new BootstrapRegistryInitializer() {
            @Override
            public void initialize(BootstrapRegistry registry) {
                registry.addCloseListener( new ApplicationListener<BootstrapContextClosedEvent>() {
                    @Override
                    public void onApplicationEvent(BootstrapContextClosedEvent event) {
                        ConfigurableListableBeanFactory beanFactory = event.getApplicationContext().getBeanFactory();
                        beanFactory.registerSingleton( "cloud", cloud() );
                        beanFactory.registerSingleton( "tracer", tracer() );
                        beanFactory.registerSingleton( "platform", platform() );
                        beanFactory.registerSingleton( "metric-registry", metricRegistry() );
                        beanFactory.registerSingleton( "health-check-registry", healthCheckRegistry() );
                    }
                } );
            }
        } );

        addInitializers( new ApplicationContextInitializer<ConfigurableApplicationContext>() {
            @Override
            public void initialize(ConfigurableApplicationContext ctx) {
                ConfigurableListableBeanFactory factory = ctx.getBeanFactory();
                factory.addBeanPostProcessor( new DestructionAwareBeanPostProcessor() {
                    @Override
                    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
                        if ( bean instanceof BootstrapAware ) {
                            try {
                                ( (BootstrapAware) bean ).setBootstrap( AbstractBootstrap.this );
                            }
                            catch ( Exception err ) {
                                throw new BeanCreationException( err.getMessage(), err );
                            }
                        }
                        return bean;
                    }
                    @Override
                    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
                        return bean;
                    }
                    @Override
                    public void postProcessBeforeDestruction(Object bean, String beanName) throws BeansException {

                    }
                } );
            }
        } );

        applicationContext = super.run( args );

        // ~ start JMX reporter
        jmxReporter.start();

        // ~ start console reporter
        if ( isDevMode() ) {}
        else {
            consoleReporter.start( props.APP_TIMER_INTERVAL.get(), TimeUnit.SECONDS );
        }

        for ( Channel acceptor : channels ) {
            applicationContext.getAutowireCapableBeanFactory().autowireBean( acceptor );
        }

        DefaultApplicationArguments arguments = new DefaultApplicationArguments( args );
        for ( Channel acceptor : channels ) {
            acceptor.run( arguments );
        }
    }
    protected void doStop() throws Exception {
        removeAllHealthChecks();

        // ~ first, stop all acceptors gracefully
        for ( Channel acceptor : channels ) {
            try {
                logger.info( "disposing acceptor {} ...", acceptor );
                acceptor.destroy();
            }
            catch ( Throwable t ) {
                logger.warn( t.getMessage(), t );
            }
        }
        try {
            if ( applicationContext != null ) {
                logger.info( "closing application now ..." );
                applicationContext.close();
            }
        }
        finally {
            //
            // ~ stop JMX reporter
            //
            logger.info( "stoppping JMX reporter ..." );
            jmxReporter.close();
            consoleReporter.close();

            //
            // ~ shutdown executor
            //
            logger.info( "shutting down platform now ..." );
            platform.dispose();

            // ~ close sentry
            if ( sentry != null ) {
                logger.info( "closing sentry now ..." );
                sentry.closeConnection();
            }

            //
            // ~ remove binders
            //
            for ( Iterator<MeterBinder> it = binders.iterator(); it.hasNext(); ) {
                MeterBinder binder = it.next();

                try {
                    if ( binder instanceof AutoCloseable ) {
                        ( (AutoCloseable) binder ).close();
                    }
                }
                catch ( Throwable t ) {
                    logger.warn( t.getMessage(), t );
                }
                finally {
                    it.remove();
                }
            }

            //
            // ~ cleanup action on configuration
            //
            if ( props.APP_CLEAR_CFG_AT_SHUTDOWN_ENABLED.get() ) {
                logger.debug( "disposing CFG ..." );

                ApplicationConfig cfg = props.cfg();
                for ( String next : cfg.getConfigNames() ) {
                    Config removed = cfg.removeConfig( next );
                    if ( removed instanceof PollingDynamicConfig ) {
                        PollingDynamicConfig pdc = (PollingDynamicConfig) removed;
                        pdc.shutdown();
                    }
                }
                cfg.shutdown();
            }
        }
    }
    private void configureLogging(MutableObject<URL> logFile) throws Exception {
        ApplicationInstanceInfo info = cloud.getApplicationInstanceInfo();
        Map<String, Object> cloudProps = info.getProperties();
        Obfuscators.init(props());

        String space = cloudProps.get( CloudOptions.CLOUD_APP_SPACE_NAME ).toString();
        String host = cloudProps.get( CloudOptions.CLOUD_APP_HOST ).toString();
        String slot = cloudProps.get( CloudOptions.CLOUD_APP_INSTANCE_INDEX ).toString();
        String service = info.getAppId();

        LoggerContext loggerFactory = (LoggerContext) LoggerFactory.getILoggerFactory();
        loggerFactory.setPackagingDataEnabled( isDevMode() );

        boolean toConfigure = true;

        Map<String, String> options = new HashMap<>();
        options.put( Logback.SPACE, space );
        options.put( Logback.HOST, host );
        options.put( Logback.SERVICE, service );
        options.put( Logback.SLOT, slot );
        options.put( Logback.RELEASE, release() );

        Map<String, Object> copyOfCfg = Maps.newHashMap();
        Iterator<String> keys = cfg().getKeys();
        while ( keys.hasNext() ) {
            String next = keys.next();
            Object raw = cfg().getRawProperty( next );
            copyOfCfg.put( next, raw );
        }

        //
        // ~ logging/alerts
        //
        AtomicBoolean logging = new AtomicBoolean( props().APP_LOGGING_DRY_RUN.get() );
        AtomicBoolean alerts = new AtomicBoolean( props().APP_ALERTS_DRY_RUN.get() );

        //
        // ~ watch GIT specifically (very important for production)
        //
        Config git = cfg().getConfig( DynamicCompositeConfig.GIT_CFG_NAME );
        git.addListener( new DefaultConfigListener() {
            @Override
            public void onConfigUpdated(Config config) {
                if ( config.containsKey( ApplicationProperties.APP_LOGGING_DRY_RUN_KEY ) ) {
                    boolean updated = config.getBoolean( ApplicationProperties.APP_LOGGING_DRY_RUN_KEY );
                    logging.set( updated );
                }
                else if ( config.containsKey( ApplicationProperties.APP_ALERTS_DRY_RUN_KEY ) ) {
                    boolean updated = config.getBoolean( ApplicationProperties.APP_ALERTS_DRY_RUN_KEY );
                    alerts.set( updated );
                }
            }
        } );

        Supplier<Boolean> loggingDryRun = new Supplier<Boolean>() {
            @Override
            public Boolean get() {
                return logging.get();
            }
        };
        Supplier<Boolean> alertsDryRun = new Supplier<Boolean>() {
            @Override
            public Boolean get() {
                return alerts.get();
            }
        };

        //
        // ~ re-load classical conventional file
        //
        if ( props().APP_DEV_MODE.get() ) {
            try {
                URL logbackTest = ResourceUtils.getURL( ResourceUtils.CLASSPATH_URL_PREFIX + "logback-test.xml" );
                try (InputStream io = logbackTest.openStream()) {
                    toConfigure = false;
                    loggerFactory.putObject( LoggingSystem.class.getName(), new Object() );
                }
            }
            catch ( FileNotFoundException err ) {

            }
        }
        if ( toConfigure ) {
            AlertLoggingFilter alertFilter = new AlertLoggingFilter( props.SENTRY_ALERTS_FILTER_ENABLED,
                                                                     props.SENTRY_ALERTS_EX_CLASSES_TO_IGNORE,
                                                                     props.SENTRY_ALERTS_EX_MSG_TO_IGNORE,
                                                                     props.SENTRY_ALERTS_EX_PATTERNS_TO_IGNORE,
                                                                     props.SENTRY_ALERTS_LOG_MESSAGES_TO_IGNORE,
                                                                     props.SENTRY_ALERTS_LOGGERS_TO_IGNORE );
            if ( logFile.getValue() != null ) {
                try (InputStream io = logFile.getValue().openStream()) {
                    Logback.configureFrom( loggerFactory, alertFilter, io, options, sentry(), copyOfCfg, loggingDryRun, alertsDryRun );
                    loggerFactory.putObject( LoggingSystem.class.getName(), new Object() ); // ~ hack
                }
            }
            else {
                try {
                    URL loggingXml = ResourceUtils.getURL( ResourceUtils.CLASSPATH_URL_PREFIX + "logging.xml" );
                    try (InputStream io = loggingXml.openStream()) {
                        Logback.configureFrom( loggerFactory, alertFilter, io, options, sentry(), copyOfCfg, loggingDryRun, alertsDryRun );
                        loggerFactory.putObject( LoggingSystem.class.getName(), new Object() ); // ~ hack
                    }
                }
                catch ( FileNotFoundException err ) {

                }
            }
        }
    }
    private void removeAllHealthChecks() {
        for ( String name : healthCheckRegistry.getNames() ) {
            logger.info( "removing health-check {} ...", name );
            healthCheckRegistry.unregister( name );
        }
    }
}
