package org.springframework.cloud;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.BiConsumer;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.InitCommand;
import org.eclipse.jgit.api.PullCommand;
import org.eclipse.jgit.lib.ConfigConstants;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.transport.RemoteConfig;
import org.eclipse.jgit.transport.TagOpt;
import org.eclipse.jgit.transport.URIish;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.slf4j.ILoggerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.app.ApplicationInstanceInfo;
import org.springframework.cloud.service.ServiceInfo;

import com.netflix.archaius.config.DefaultSettableConfig;
import com.turbospaces.boot.AbstractBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.CloudMapConfig;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.cfg.DynamicCompositeConfig;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.logging.Logback;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.UPSs;

public class ConfigurableCloudFactory extends CloudFactory {
    private final Logger logger = LoggerFactory.getLogger( getClass() );

    private final AbstractBootstrap bootstrap;
    private final Cloud cloud;
    private Optional<String> loggingXmlAsString = Optional.empty();

    public ConfigurableCloudFactory(ConfigurableCloudConnector connector, AbstractBootstrap bootstrap) throws Exception {
        registerCloudConnector( connector );

        this.bootstrap = Objects.requireNonNull( bootstrap );
        this.cloud = super.getCloud();

        // ~ schedule dynamic reload
        ApplicationConfig cfg = bootstrap.props().cfg();
        Optional<ServiceInfo> opt = UPSs.findServiceInfoByName( cloud, UPSs.CFG );
        if ( opt.isPresent() ) {
            if ( bootstrap.props().APP_CFG_DYNAMIC_POLLING_ENABLED.get() ) {
                // periodically poll for changes
                cfg.execute( new Runnable() {
                    @Override
                    public void run() {
                        try {
                            apply();
                        }
                        catch ( Exception ex ) {
                            logger.error( ex.getMessage(), ex );
                        }
                    }
                } );
            }
            else {
                apply();
            }
        }
        else {
            logger.debug( "'{}' is not bound to app ...", UPSs.CFG );
        }

        // ~ replace configuration with real implementation
        CloudMapConfig cfc = new CloudMapConfig( connector.getApplicationInstanceInfo() );
        DefaultSettableConfig config = (DefaultSettableConfig) cfg.getConfig( DynamicCompositeConfig.CLOUD_CFG_NAME );
        config.setProperties( cfc );

        // ~ configure logging framework now
        configureLogging();
    }
    @Override
    public Cloud getCloud() {
        return cloud; // ~ safe to call many times
    }
    private void configureLogging() throws Exception {
        ApplicationInstanceInfo info = cloud.getApplicationInstanceInfo();
        Map<String, Object> cloudProps = info.getProperties();

        String space = cloudProps.get( CloudOptions.CLOUD_APP_SPACE_NAME ).toString();
        String host = cloudProps.get( CloudOptions.CLOUD_APP_HOST ).toString();
        String slot = cloudProps.get( CloudOptions.CLOUD_APP_INSTANCE_INDEX ).toString();
        String service = info.getAppId();
        String release = PlatformUtil.version( bootstrap.props().CLOUD_APP_NAME );

        ILoggerFactory iLoggerFactory = LoggerFactory.getILoggerFactory();
        boolean toConfigure = true;

        Map<String, String> map = new HashMap<>();
        map.put( Logback.SPACE, space );
        map.put( Logback.HOST, host );
        map.put( Logback.SERVICE, service );
        map.put( Logback.SLOT, slot );
        map.put( Logback.RELEASE, release );
        map.put( Logback.LOGGING_DRY_RUN, Boolean.toString( bootstrap.props().APP_LOGGING_DRY_RUN.get() ) );
        map.put( Logback.ALERTS_DRY_RUN, Boolean.toString( bootstrap.props().APP_ALERTS_DRY_RUN.get() ) );

        if ( bootstrap.props().APP_DEV_MODE.get() ) {
            Optional<URL> logbackTest = PlatformUtil.getResource( "logback-test.xml" );
            if ( logbackTest.isPresent() ) {
                try (InputStream io = logbackTest.get().openStream()) {
                    toConfigure = false;
                }
            }
        }
        if ( toConfigure ) {
            if ( loggingXmlAsString.isPresent() ) {
                String xmlAsString = loggingXmlAsString.get();
                byte[] xmlAsBytes = xmlAsString.getBytes( StandardCharsets.UTF_8 );
                try (InputStream io = new ByteArrayInputStream( xmlAsBytes )) {
                    Logback.configureFrom( iLoggerFactory, io, map, bootstrap.sentry() );
                }
            }
            else {
                Optional<URL> loggingXml = PlatformUtil.getResource( "logging.xml" );
                if ( loggingXml.isPresent() ) {
                    try (InputStream io = loggingXml.get().openStream()) {
                        Logback.configureFrom( iLoggerFactory, io, map, bootstrap.sentry() );
                    }
                }
            }
        }
    }
    private void apply() throws Exception {
        ApplicationConfig cfg = bootstrap.props().cfg();
        Optional<ServiceInfo> opt = UPSs.findServiceInfoByName( cloud, UPSs.CFG );
        if ( opt.isPresent() ) {
            PlainServiceInfo info = (PlainServiceInfo) opt.get();
            DefaultSettableConfig gitCfg = (DefaultSettableConfig) cfg.getConfig( DynamicCompositeConfig.GIT_CFG_NAME );

            File toFile = Files.createTempDirectory( "cfg" ).toFile();
            InitCommand initCmd = new InitCommand().setBare( false ).setDirectory( toFile );
            logger.debug( "cloning git repo from={}", info );

            try (Git git = initCmd.call()) {
                applyConfig( git, info );

                // pull
                PullCommand pullCommand = git.pull().setRemote( Constants.DEFAULT_REMOTE_NAME ).setTagOpt( TagOpt.FETCH_TAGS );
                pullCommand.setRemoteBranchName( Constants.MASTER );
                if ( StringUtils.isNotEmpty( info.getUserName() ) && StringUtils.isNotEmpty( info.getPassword() ) ) {
                    pullCommand.setCredentialsProvider( new UsernamePasswordCredentialsProvider( info.getUserName(), info.getPassword() ) );
                }
                pullCommand.call();

                Map<String, Object> cloudProps = cloud.getApplicationInstanceInfo().getProperties();
                String spaceName = cloudProps.get( CloudOptions.CLOUD_APP_SPACE_NAME ).toString();
                String appId = cloud.getApplicationInstanceInfo().getAppId();
                String filename = "application.properties";

                Properties gitProps = new Properties();
                Runnable task = new Runnable() {
                    @Override
                    public void run() {
                        Set<String> toRemove = new HashSet<>();
                        gitCfg.forEachProperty( new BiConsumer<String, Object>() {
                            @Override
                            public void accept(String k, Object v) {
                                if ( gitProps.containsKey( k ) ) {}
                                else {
                                    toRemove.add( k );
                                }
                            }
                        } );
                        if ( toRemove.isEmpty() ) {}
                        else {
                            logger.debug( "found orphaned properties: {}", toRemove );
                            for ( String prop : toRemove ) {
                                gitCfg.clearProperty( prop );
                            }
                        }
                        logger.debug( "setting git cfg props={}", gitProps );
                        cfg.setGitProperties( gitProps );
                    }
                };

                logger.debug( "looking for '{}' in {}:{}", filename, spaceName, appId );
                File dspace = new File( toFile, spaceName );
                if ( dspace.exists() && dspace.isDirectory() ) {
                    File fspace = new File( dspace, filename );

                    // ~ try load space shared properties
                    if ( fspace.exists() && fspace.isFile() ) {
                        try (FileInputStream io = new FileInputStream( fspace )) {
                            gitProps.load( io );

                        }
                    }

                    File dapp = new File( dspace, appId );
                    if ( dapp.exists() && dspace.isDirectory() ) {
                        File fapp = new File( dapp, filename );
                        if ( fapp.exists() && fapp.isFile() ) {
                            try (FileInputStream io = new FileInputStream( fapp )) {
                                gitProps.load( io );
                            }
                        }

                        filename = "logging.xml";
                        fapp = new File( dapp, filename );
                        if ( fapp.exists() && fapp.isFile() ) {
                            String xml = new String( Files.readAllBytes( fapp.toPath() ), StandardCharsets.UTF_8 );
                            loggingXmlAsString = Optional.of( xml );
                        }
                    }
                }

                task.run(); // ~ set properties
            }
            finally {
                if ( toFile.exists() ) {
                    toFile.delete();
                }
            }
        }
    }
    private void applyConfig(Git git, PlainServiceInfo info) throws URISyntaxException, IOException {
        logger.debug( "patching git config ..." );
        StoredConfig config = git.getRepository().getConfig();
        config.setBoolean( "http", null, "sslVerify", false );
        config.setBoolean( ConfigConstants.CONFIG_GC_SECTION, null, ConfigConstants.CONFIG_KEY_AUTODETACH, false );

        URIish urish = new URIish( StringUtils.removeEnd( info.getUri(), "/" ) );
        RemoteConfig remoteConfig = new RemoteConfig( config, Constants.DEFAULT_REMOTE_NAME );
        remoteConfig.addURI( urish );

        RefSpec refSpec = new RefSpec();
        refSpec = refSpec.setForceUpdate( true );
        refSpec = refSpec.setSourceDestination( Constants.R_HEADS + "*", Constants.R_REMOTES + remoteConfig.getName() + "/" + "*" );
        remoteConfig.addFetchRefSpec( refSpec );
        remoteConfig.update( config );

        config.save();
    }
}
