package com.turbospaces.di;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.inject.Provider;

import org.apache.commons.lang3.concurrent.ConcurrentException;
import org.apache.commons.lang3.concurrent.ConcurrentInitializer;
import org.apache.commons.lang3.concurrent.LazyInitializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.inject.Binder;
import com.google.inject.ProvisionException;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.boot.BootstrapAware;
import com.turbospaces.boot.BootstrapPlugin;
import com.turbospaces.cfg.ApplicationProperties;

public abstract class AbstractBootstrapPlugin<T> implements BootstrapAware, BootstrapPlugin, ConcurrentInitializer<T>, Provider<T>, PreDestroyable {
    protected final Logger logger = LoggerFactory.getLogger( getClass() );
    protected Bootstrap bootstrap;

    private final AtomicBoolean destroyed = new AtomicBoolean();
    private final ConcurrentInitializer<T> init;
    private final Class<T> type;

    protected AbstractBootstrapPlugin(Class<T> type) {
        this.type = Objects.requireNonNull( type );
        this.init = new LazyInitializer<T>() {
            @Override
            protected T initialize() throws ConcurrentException {
                try {
                    return createOnce();
                }
                catch ( RuntimeException err ) {
                    throw err;
                }
                catch ( Exception err ) {
                    throw new ConcurrentException( err );
                }
            }
        };
    }
    @Override
    public void setBootstrap(Bootstrap bootstrap) throws Exception {
        this.bootstrap = Objects.requireNonNull( bootstrap );
    }
    @Override
    public Bootstrap getBootstrap() {
        return bootstrap;
    }
    @Override
    public final void preDestroy() throws Exception {
        if ( destroyed.compareAndSet( false, true ) ) {
            T obj = init.get();
            if ( obj != null ) {
                destroyOnce( obj );
            }
        }
    }
    @Override
    public final T get() {
        try {
            return init.get();
        }
        catch ( Exception err ) {
            throw new ProvisionException( "failed to configure plugin", err );
        }
    }
    @Override
    public final void configure(Binder binder) {
        binder.requestInjection( this );
        binder.bind( type ).toProvider( this ).asEagerSingleton();
    }
    protected ApplicationProperties props() {
        return bootstrap.props();
    }
    protected abstract T createOnce() throws Exception;
    protected abstract void destroyOnce(T obj) throws Exception;
}
