package com.turbospaces.boot;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.time.LocalDate;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Properties;

import org.apache.commons.lang3.RandomStringUtils;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.common.CassandraServiceInfo;
import org.springframework.cloud.service.common.RedisServiceInfo;

import com.google.common.io.ByteSource;
import com.google.common.net.HostAndPort;
import com.netflix.archaius.api.exceptions.ConfigException;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.ups.H2DatabaseServiceInfo;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.RawServiceInfo;
import com.turbospaces.ups.UPSs;
import com.turbospaces.ups.ZookeeperServiceInfo;

public class MockCloud {
    private final Properties cloudPropsOverride = new Properties();
    private final ApplicationConfig cfg;
    private final List<ServiceInfo> services = new LinkedList<>();
    private final String spaceName;
    private final String appId;
    private final String appName;

    private MockCloud(String appId, String appName) throws ConfigException {
        this.appId = Objects.requireNonNull( appId );
        this.appName = Objects.requireNonNull( appName );
        this.spaceName = RandomStringUtils.randomAlphabetic( 4 );
        this.cfg = ApplicationConfig.create();
    }
    public MockCloud withKafka(int port) {
        return withKafka( HostAndPort.fromParts( "localhost", port ) );
    }
    public MockCloud withKafka(HostAndPort addr) {
        PlainServiceInfo info = UPSs.toKafkaServiceInfo( addr );
        addUserProvidedService( info );
        return this;
    }
    public MockCloud withRedis(int port) {
        return withRedis( HostAndPort.fromParts( "localhost", port ) );
    }
    public MockCloud withRedis(HostAndPort addr) {
        RedisServiceInfo info = new RedisServiceInfo( UPSs.REDIS, addr.getHost(), addr.getPort(), "na" );
        addUserProvidedService( info );
        return this;
    }
    public MockCloud withZookeeper(int port) {
        return withZookeeper( HostAndPort.fromParts( "localhost", port ) );
    }
    public MockCloud withZookeeper(HostAndPort addr) {
        ZookeeperServiceInfo info = UPSs.toZookeeperServiceInfo( addr );
        addUserProvidedService( info );
        return this;
    }
    public MockCloud withWeb3j(int port) {
        return withWeb3j( HostAndPort.fromParts( "localhost", port ) );
    }
    public MockCloud withWeb3j(HostAndPort addr) {
        PlainServiceInfo info = UPSs.toWeb3jServiceInfo( addr );
        addUserProvidedService( info );
        return this;
    }
    public MockCloud withWeb3Credentials(String password, File folder) {
        String path = String.format( "file://api_key:%s%s", password, folder.getAbsolutePath() );
        PlainServiceInfo info = new PlainServiceInfo( UPSs.WEB3J_CREDENTIALS, path );
        addUserProvidedService( info );
        return this;
    }
    public MockCloud withCassandra(int port) {
        return withCassandra( HostAndPort.fromParts( "localhost", port ) );
    }
    public MockCloud withRawService(String id, URL url) throws IOException {
        try (InputStream io = url.openStream()) {
            RawServiceInfo serviceInfo = new RawServiceInfo( id, new ByteSource() {
                @Override
                public InputStream openStream() throws IOException {
                    return io;
                }
            }.read() );
            addUserProvidedService( serviceInfo );
            return this;
        }
    }
    public MockCloud withCassandra(HostAndPort addr) {
        CassandraServiceInfo info = UPSs.toCassandraServiceInfo( addr );
        addUserProvidedService( info );
        return this;
    }
    public MockCloud withH2(boolean durable) {
        String url = "h2://mem:" + spaceName;
        if ( durable ) {
            url += ";" + "DB_CLOSE_DELAY=-1";
        }
        H2DatabaseServiceInfo ownerInfo = new H2DatabaseServiceInfo( UPSs.H2_OWNER, url );
        H2DatabaseServiceInfo appInfo = new H2DatabaseServiceInfo( UPSs.H2_APP, url );
        addUserProvidedService( ownerInfo );
        addUserProvidedService( appInfo );
        return this;
    }
    public MockCloud appPort(int port) {
        cloudPropsOverride.put( CloudOptions.CLOUD_APP_PORT, port );
        return this;
    }
    public MockCloud addUserProvidedService(ServiceInfo info) {
        services.add( info );
        return this;
    }
    public ApplicationConfig build() throws Exception {
        String slot = "0";
        int port = PlatformUtil.freePort();
        String hostname = PlatformUtil.detectIp();

        // mock typical cloud properties
        Properties props = new Properties();
        props.put( CloudOptions.CLOUD_APP_ID, appId );
        props.put( CloudOptions.CLOUD_APP_SPACE_NAME, spaceName );
        props.put( CloudOptions.CLOUD_APP_HOST, hostname );
        props.put( CloudOptions.CLOUD_APP_PORT, port );
        props.put( CloudOptions.CLOUD_APP_INSTANCE_INDEX, slot );
        props.put( CloudOptions.CLOUD_APP_NAME, appName );
        props.putAll( cloudPropsOverride );

        // override local properties
        cfg.setLocalProperty( ApplicationProperties.APP_SHUTDOWN_HOOK_ENABLED_KEY, false );
        cfg.setLocalProperties( UPSs.addUserProvideServices( props, services ) );

        //
        return cfg;
    }
    public static MockCloud fromClasspathAppProps() throws Exception {
        // load required properties file
        URL resource = PlatformUtil.getRequiredResource( "application.properties" );
        return fromClasspathAppProps( resource );
    }
    public static MockCloud fromClasspathAppProps(URL url) throws Exception {
        // load required properties file
        try (InputStream io = url.openStream()) {
            Properties props = new Properties();
            props.load( io );
            String appId = props.getProperty( CloudOptions.CLOUD_APP_ID );
            String appName = props.getProperty( CloudOptions.CLOUD_APP_NAME, randomAppName() );
            return new MockCloud( appId, appName );
        }
    }
    public static MockCloud newMock() throws ConfigException {
        int length = MockCloud.class.getSimpleName().length();
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric( length );
        return new MockCloud( randomAppId, randomAppName() );
    }
    private static String randomAppName() {
        int length = MockCloud.class.getSimpleName().length();
        LocalDate now = LocalDate.now();
        String randomGroup = "group-" + RandomStringUtils.randomAlphanumeric( length );
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric( length );
        String randomVersion = now.getYear() + "" + now.getMonthValue() + "" + now.getDayOfMonth();
        String randomAppName = String.format( "%s:%s:%s", randomGroup, randomAppId, randomVersion );
        return randomAppName;
    }
}
