package com.turbospaces.boot;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor.CallerRunsPolicy;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.common.ThrowableAction;

import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.Schedulers;

public class FixedSizePlatform implements Platform {
    private final Logger logger = LoggerFactory.getLogger( getClass() );
    private final Scheduler scheduler;
    private final ScheduledThreadPoolExecutor executor;
    private final ApplicationProperties props;

    public FixedSizePlatform(ApplicationProperties props, MeterRegistry meter) {
        this.props = Objects.requireNonNull( props );

        int psize = props.APP_PLATFORM_POOL_SIZE.get();

        ThreadFactoryBuilder tfb = new ThreadFactoryBuilder();
        tfb.setDaemon( false );
        tfb.setNameFormat( "platform-worker-%d" );

        executor = new ScheduledThreadPoolExecutor( psize, tfb.build(), new CallerRunsPolicy() );
        executor.prestartAllCoreThreads();

        scheduler = Schedulers.fromExecutorService( executor, "reactor" );

        ExecutorServiceMetrics metrics = new ExecutorServiceMetrics( executor, "platform", Collections.emptyList() );
        metrics.bindTo( meter );
    }
    @Override
    public void shutdown() {
        executor.shutdown();
    }
    @Override
    public List<Runnable> shutdownNow() {
        return executor.shutdownNow();
    }
    @Override
    public boolean isShutdown() {
        return executor.isShutdown();
    }
    @Override
    public boolean isTerminated() {
        return executor.isTerminated();
    }
    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return executor.awaitTermination( timeout, unit );
    }
    @Override
    public <T> Future<T> submit(Callable<T> task) {
        return executor.submit( task );
    }
    @Override
    public <T> Future<T> submit(Runnable task, T result) {
        return executor.submit( task, result );
    }
    @Override
    public Future<?> submit(Runnable task) {
        return executor.submit( task );
    }
    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
        return executor.invokeAll( tasks );
    }
    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException {
        return executor.invokeAll( tasks, timeout, unit );
    }
    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
        return executor.invokeAny( tasks );
    }
    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        return executor.invokeAny( tasks, timeout, unit );
    }
    @Override
    public ScheduledFuture<?> schedule(Runnable command, long delay, TimeUnit unit) {
        return executor.schedule( command, delay, unit );
    }
    @Override
    public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit) {
        return executor.schedule( callable, delay, unit );
    }
    @Override
    public ScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit) {
        return executor.scheduleAtFixedRate( command, initialDelay, period, unit );
    }
    @Override
    public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, long initialDelay, long delay, TimeUnit unit) {
        return executor.scheduleWithFixedDelay( command, initialDelay, delay, unit );
    }
    @Override
    public Scheduler scheduler() {
        return scheduler;
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdc = MDC.getCopyOfContextMap(); // ~ capture MDC
        if ( mdc == null ) {
            executor.execute( command );
        }
        else {
            executor.execute( new Runnable() {
                @Override
                public void run() {
                    MDC.setContextMap( mdc ); // ~ set MDC stuff back if necessary
                    try {
                        command.run();
                    }
                    catch ( Throwable err ) {
                        logger.error( err.getMessage(), err );
                    }
                    finally {
                        MDC.clear(); // ~ finally clear it if necessary
                    }
                }
            } );
        }
    }
    @Override
    public void work(ThrowableAction action) {
        execute( new Runnable() {
            @Override
            public void run() {
                try {
                    action.apply();
                }
                catch ( Exception err ) {
                    logger.error( err.getMessage(), err );
                }
            }
        } );
    }
    @PreDestroy
    @Override
    public void preDestroy() {
        int timeout = props.APP_PLATFORM_GRACEFUL_TIMEOUT.get();
        PlatformUtil.shutdownExecutor( executor, timeout );
        logger.debug( "stopped platform-worker pool ..." );
    }
}
