package com.turbospaces.boot;

import java.lang.reflect.UndeclaredThrowableException;
import java.security.KeyStore;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;

import org.apache.commons.lang3.mutable.MutableObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.Cloud;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.ThrowableAction1;
import com.turbospaces.di.DiEngine;

import io.micrometer.core.instrument.MeterRegistry;
import io.opentracing.Tracer;
import io.sentry.SentryClient;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.Schedulers;
import reactor.util.retry.Retry;

public interface Bootstrap {
    //
    // life-cycle
    //
    void start(DiEngine engine) throws Exception;
    void shutdown() throws Exception;

    //
    // meta
    //
    ApplicationStatus status();
    Date startedAt();

    boolean isDevMode();
    String release();
    String spaceName();
    String appId();

    KeyStore keyStore();

    //
    // main port
    //
    int port();

    //
    // health-checks and metrics support
    //
    MetricRegistry metricRegistry();
    MeterRegistry meterRegisry();
    HealthCheckRegistry healthCheckRegistry();
    SentryClient sentry();
    Tracer tracer();

    //
    // thread-pool / configuration / cloud / DI engine
    //
    Platform platform();
    ApplicationProperties props();
    Cloud cloud();
    DiEngine diEngine();

    //
    // plugins + channels
    //
    boolean addPlugin(BootstrapPlugin plugin);
    boolean addChannel(Channel acceptor);
    Collection<BootstrapPlugin> plugins();
    Collection<Channel> channels();

    //
    // helper-methods
    //
    boolean isHealthy();
    void registerHealthCheck(String name, HealthCheck check);

    default Retry retry() {
        return props().retry( platform().scheduler() );
    }
    default <V> void retry(V value, ThrowableAction1<V> action) {
        retry( value, action, platform().scheduler() );
    }
    default <V> void retry(V value, ThrowableAction1<V> action, Scheduler scheduler) {
        Logger logger = LoggerFactory.getLogger( getClass() );
        AtomicLong seq = new AtomicLong();

        Mono.just( value ).doOnError( new Consumer<Throwable>() {
            @Override
            public void accept(Throwable t) {
                logger.error( t.getMessage(), t );
            }
        } ).doOnNext( new Consumer<V>() {
            @Override
            public void accept(V target) {
                try {
                    action.apply( target );
                }
                catch ( Exception err ) {
                    logger.warn( "got failure on iteration: " + seq.getAndIncrement(), err );
                    throw new UndeclaredThrowableException( err );
                }
            }
        } ).retryWhen( props().retry( scheduler ) ).block();
    }
    default <V> void retry(Flux<V> stream, ThrowableAction1<List<V>> action) {
        retry( stream, action, platform().scheduler() );
    }
    default <V> void retry(Flux<V> stream, ThrowableAction1<List<V>> action, Scheduler scheduler) {
        Logger logger = LoggerFactory.getLogger( getClass() );
        AtomicLong seq = new AtomicLong();

        stream.collectList().doOnError( new Consumer<Throwable>() {
            @Override
            public void accept(Throwable t) {
                logger.error( t.getMessage(), t );
            }
        } ).doOnNext( new Consumer<List<V>>() {
            @Override
            public void accept(List<V> target) {
                try {
                    action.apply( target );
                }
                catch ( Exception err ) {
                    logger.warn( "got failure on iteration: " + seq.getAndIncrement(), err );
                    throw new UndeclaredThrowableException( err );
                }
            }
        } ).retryWhen( props().retry( scheduler ) ).block();
    }
    default <V> V retryCallable(Callable<V> callback) {
        return retryCallable( callback, Schedulers.immediate() );
    }
    default <V> V retryCallable(Callable<V> callback, Scheduler scheduler) {
        Logger logger = LoggerFactory.getLogger( getClass() );
        MutableObject<V> result = new MutableObject<V>();
        AtomicLong seq = new AtomicLong();

        Mono.just( callback ).doOnError( new Consumer<Throwable>() {
            @Override
            public void accept(Throwable t) {
                logger.error( t.getMessage(), t );
            }
        } ).doOnNext( new Consumer<Callable<V>>() {
            @Override
            public void accept(Callable<V> target) {
                try {
                    V value = target.call();
                    result.setValue( value );
                }
                catch ( Exception err ) {
                    logger.warn( "got failure on iteration: " + seq.getAndIncrement(), err );
                    throw new UndeclaredThrowableException( err );
                }
            }
        } ).retryWhen( props().retry( scheduler ) ).block();

        return result.getValue();
    }
}
