package com.turbospaces.boot;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.security.KeyStore;
import java.security.Security;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.Cloud;
import org.springframework.cloud.ConfigurableCloudConnector;
import org.springframework.cloud.ConfigurableCloudFactory;
import org.springframework.cloud.app.ApplicationInstanceInfo;
import org.springframework.cloud.service.UriBasedServiceInfo;

import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Slf4jReporter;
import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheck.Result;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.codahale.metrics.health.HealthCheckRegistryListener;
import com.codahale.metrics.jmx.JmxReporter;
import com.codahale.metrics.jvm.ThreadDump;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.Uninterruptibles;
import com.google.inject.Injector;
import com.netflix.archaius.api.Config;
import com.netflix.archaius.config.PollingDynamicConfig;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.di.DiEngine;
import com.turbospaces.di.PostConstructable;
import com.turbospaces.di.PreDestroyable;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.UPSs;

import io.jaegertracing.Configuration;
import io.jaegertracing.internal.Constants;
import io.jaegertracing.internal.JaegerTracer;
import io.micrometer.core.instrument.Clock;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.binder.jvm.JvmMemoryMetrics;
import io.micrometer.core.instrument.binder.system.UptimeMetrics;
import io.micrometer.core.instrument.composite.CompositeMeterRegistry;
import io.micrometer.core.instrument.dropwizard.DropwizardConfig;
import io.micrometer.core.instrument.dropwizard.DropwizardMeterRegistry;
import io.micrometer.core.instrument.util.HierarchicalNameMapper;
import io.micrometer.elastic.ElasticConfig;
import io.micrometer.elastic.ElasticMeterRegistry;
import io.micrometer.influx.InfluxConfig;
import io.micrometer.influx.InfluxMeterRegistry;
import io.opentracing.Tracer;
import io.sentry.SentryClient;
import io.sentry.SentryClientFactory;
import io.sentry.connection.EventSendCallback;
import io.sentry.event.Event;
import io.sentry.event.helper.ShouldSendEventCallback;

public abstract class AbstractBootstrap implements Bootstrap, ShouldSendEventCallback {
    static {
        Security.addProvider( new BouncyCastleProvider() );
    }

    private final Logger logger = LoggerFactory.getLogger( getClass() );
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final MetricRegistry metricRegistry = new MetricRegistry();
    private final CompositeMeterRegistry meterRegistry = Metrics.globalRegistry;
    private final Set<Channel> channels = new LinkedHashSet<>();
    private final Set<BootstrapPlugin> plugins = new LinkedHashSet<>();

    private final JmxReporter jmxReporter;
    private final HealthCheckRegistry healthCheckRegistry;
    private final Thread shutdownHook;
    private final String release;
    private final Cloud cloud;
    private final ApplicationProperties props;
    private final FixedSizePlatform platform;
    private final SentryClient sentry;
    private final JaegerTracer tracer;

    private KeyStore keyStore;
    private ApplicationStatus status = ApplicationStatus.UNKNOWN;
    private Injector parent;
    private DiEngine diEngine;
    private Date started;

    protected AbstractBootstrap(ApplicationProperties props) throws Exception {
        this.props = Objects.requireNonNull( props );
        this.keyStore = KeyStore.getInstance( KeyStore.getDefaultType() );

        ConfigurableCloudConnector connector = new ConfigurableCloudConnector( props, keyStore );
        ApplicationInstanceInfo info = connector.getApplicationInstanceInfo();
        Map<String, Object> cloudProps = info.getProperties();

        String space = cloudProps.get( CloudOptions.CLOUD_APP_SPACE_NAME ).toString();
        String host = cloudProps.get( CloudOptions.CLOUD_APP_HOST ).toString();
        String slot = cloudProps.get( CloudOptions.CLOUD_APP_INSTANCE_INDEX ).toString();
        String service = info.getAppId();
        release = PlatformUtil.version( props.CLOUD_APP_NAME );

        // ~ sentry
        Optional<UriBasedServiceInfo> sentryOpt = UPSs.findServiceInfoByName( connector, UPSs.SENTRY );
        if ( sentryOpt.isPresent() ) {
            UriBasedServiceInfo serviceInfo = sentryOpt.get();
            sentry = SentryClientFactory.sentryClient( serviceInfo.getUri() );
        }
        else {
            sentry = SentryClientFactory.sentryClient();
        }
        sentry.setEnvironment( space );
        sentry.setRelease( release );
        sentry.setServerName( service + "/" + slot );
        sentry.addShouldSendEventCallback( this );
        sentry.addEventSendCallback( new EventSendCallback() {
            @Override
            public void onSuccess(Event event) {
                Counter counter = metricRegistry.counter( MetricRegistry.name( "sentry", "success" ) );
                counter.inc();
            }
            @Override
            public void onFailure(Event event, Exception exception) {
                Counter counter = metricRegistry.counter( MetricRegistry.name( "sentry", "failure" ) );
                counter.inc();
            }
        } );

        // ~ cloud
        cloud = new ConfigurableCloudFactory( connector, this ).getCloud();

        // ~ tracer
        ImmutableMap.Builder<String, String> tracingTags = ImmutableMap.builder();
        tracingTags.put( Constants.TRACER_IP_TAG_KEY, PlatformUtil.detectIp() );
        tracingTags.put( Constants.TRACER_HOSTNAME_TAG_KEY, host );
        tracingTags.put( CloudOptions.CLOUD_APP_INSTANCE_INDEX, slot );

        Configuration fromEnv = Configuration.fromEnv( service ).withTracerTags( tracingTags.build() );
        tracer = fromEnv.getTracer();

        ImmutableList.Builder<Tag> meterTags = ImmutableList.builder();
        meterTags.add( Tag.of( "env", space ) );
        meterTags.add( Tag.of( "release", release ) );
        meterTags.add( Tag.of( "service", service ) );
        meterRegistry.config().commonTags( meterTags.build() );

        meterRegistry.add( new DropwizardMeterRegistry( new DropwizardConfig() {
            @Override
            public String prefix() {
                return "boot";
            }
            @Override
            public String get(String key) {
                return props.cfg().getString( key, null );
            }
        }, metricRegistry, HierarchicalNameMapper.DEFAULT, Clock.SYSTEM ) {
            @Override
            protected Double nullGaugeValue() {
                return Double.NaN;
            }
        } );

        if ( props.APP_METRICS_DRY_RUN.get() ) {}
        else {
            Optional<PlainServiceInfo> optelk = UPSs.findServiceInfoByName( this, UPSs.ELASTIC_SEARCH );
            Optional<PlainServiceInfo> optinf = UPSs.findServiceInfoByName( this, UPSs.INFLUX );

            if ( optelk.isPresent() ) {
                PlainServiceInfo si = optelk.get();

                if ( props.APP_METRICS_ELK_REPORTER_ENABLED.get() ) {
                    meterRegistry.add( new ElasticMeterRegistry( new ElasticConfig() {
                        @Override
                        public int batchSize() {
                            return props.APP_METRICS_BULK_SIZE.get();
                        }
                        @Override
                        public String host() {
                            return String.format( "%s://%s:%d", si.getScheme(), si.getHost(), si.getPort() );
                        }
                        @Override
                        public String userName() {
                            return si.getUserName();
                        }
                        @Override
                        public String password() {
                            return si.getPassword();
                        }
                        @Override
                        public Duration connectTimeout() {
                            return Duration.ofSeconds( props.TCP_CONNECTION_TIMEOUT.get() );
                        }
                        @Override
                        public Duration readTimeout() {
                            return Duration.ofSeconds( props.TCP_SOCKET_TIMEOUT.get() );
                        }
                        @Override
                        public String get(String k) {
                            return props.cfg().getString( k, null );
                        }
                    }, Clock.SYSTEM ) );
                }
            }

            if ( optinf.isPresent() ) {
                PlainServiceInfo si = optinf.get();

                if ( props.APP_METRICS_INFLUX_REPORTER_ENABLED.get() ) {
                    meterRegistry.add( new InfluxMeterRegistry( new InfluxConfig() {
                        @Override
                        public int batchSize() {
                            return props.APP_METRICS_BULK_SIZE.get();
                        }
                        @Override
                        public String uri() {
                            return String.format( "%s://%s:%d", si.getScheme(), si.getHost(), si.getPort() );
                        }
                        @Override
                        public String userName() {
                            return si.getUserName();
                        }
                        @Override
                        public String password() {
                            return si.getPassword();
                        }
                        @Override
                        public Duration connectTimeout() {
                            return Duration.ofSeconds( props.TCP_CONNECTION_TIMEOUT.get() );
                        }
                        @Override
                        public Duration readTimeout() {
                            return Duration.ofSeconds( props.TCP_SOCKET_TIMEOUT.get() );
                        }
                        @Override
                        public String get(String k) {
                            return props.cfg().getString( k, null );
                        }
                    }, Clock.SYSTEM ) );
                }
            }
        }

        JvmMemoryMetrics jvmMetrics = new JvmMemoryMetrics();
        jvmMetrics.bindTo( meterRegistry );

        UptimeMetrics uptimeMetrics = new UptimeMetrics();
        uptimeMetrics.bindTo( meterRegistry );

        // ~ JMX
        jmxReporter = JmxReporter.forRegistry( metricRegistry ).inDomain( props.APP_JMX_DOMAIN.get() ).build();

        // ~ thread pool
        platform = new FixedSizePlatform( props, meterRegistry );

        // ~ shutdown hook
        shutdownHook = new Thread( new Runnable() {
            @Override
            public void run() {
                logger.info( "running shutdown hook now ..." );
                try {
                    shutdown();
                }
                catch ( Throwable err ) {
                    logger.error( err.getMessage(), err );
                }
            }
        } );

        // ~ health check registry
        healthCheckRegistry = new HealthCheckRegistry();
        healthCheckRegistry.addListener( new HealthCheckRegistryListener() {
            @Override
            public void onHealthCheckAdded(String name, HealthCheck healthCheck) {
                if ( healthCheck instanceof BootstrapAware ) {
                    try {
                        ( (BootstrapAware) healthCheck ).setBootstrap( AbstractBootstrap.this );
                    }
                    catch ( Exception err ) {
                        ExceptionUtils.wrapAndThrow( err );
                    }
                }
                // ~ inject members if possible (check added after plug-in creation)
                if ( parent != null ) {
                    parent.injectMembers( healthCheck );
                }
            }
            @Override
            public void onHealthCheckRemoved(String name, HealthCheck healthCheck) {
                if ( healthCheck instanceof DisposableHealtchCheck ) {
                    DisposableHealtchCheck preDestoy = (DisposableHealtchCheck) healthCheck;
                    try {
                        preDestoy.preDestroy();
                    }
                    catch ( Exception err ) {
                        logger.error( err.getMessage(), err );
                    }
                }
            }
        } );
    }
    @Override
    public final void start(DiEngine engine) throws Exception {
        boolean isHealthy = true;
        Set<String> unhealthy = new HashSet<>();

        Lock rwLock = lock.writeLock();
        rwLock.lock();
        try {
            this.status = ApplicationStatus.STARTING;
            this.diEngine = engine;

            if ( props.APP_WAIT_FOR_HEALTHCHECKS_ENABLED.get() ) {
                logger.debug( "about to run health-checks now ..." );

                int it = 0;
                long now = System.currentTimeMillis();
                long timeout = TimeUnit.SECONDS.toMillis( props.APP_WAIT_FOR_HEALTHCHECKS_TIMEOUT.get() );
                isHealthy = false;

                while ( ( System.currentTimeMillis() - now ) <= timeout ) {
                    boolean tmp = true;
                    it++;
                    unhealthy.clear();
                    for ( Entry<String, Result> entry : healthCheckRegistry().runHealthChecks().entrySet() ) {
                        logger.debug( "iteration({}) ::: healthcheck({}) - isHealthy({})", it, entry.getKey(), entry.getValue().isHealthy() );
                        tmp &= entry.getValue().isHealthy();
                        if ( entry.getValue().isHealthy() ) {}
                        else {
                            unhealthy.add( entry.getKey() );
                        }
                    }
                    if ( tmp ) {
                        isHealthy = true;
                        break;
                    }
                    int waitSec = props.APP_WAIT_FOR_HEALTHCHECKS_INTERVAL.get();
                    logger.debug( "about to wait {} sec before next health_check attempt ...", waitSec );
                    Uninterruptibles.sleepUninterruptibly( waitSec, TimeUnit.SECONDS );
                }
            }

            if ( isHealthy ) {
                long now = System.currentTimeMillis();

                logger.debug( "about to perform actual application start ..." );
                doStart();
                logger.info( "application started in={} sec ...", TimeUnit.MILLISECONDS.toSeconds( System.currentTimeMillis() - now ) );

                this.started = new Date();
                this.status = ApplicationStatus.RUNNING;

                // register shutdown hook
                if ( props.APP_SHUTDOWN_HOOK_ENABLED.get() ) {
                    logger.info( "shutdown hook has been registered ..." );
                    Runtime.getRuntime().addShutdownHook( shutdownHook );
                }
            }
            else {
                logger.info( "app will not start due to failed healch_checks ..." );
                removeAllHealthChecks();
                if ( platform != null ) {
                    platform.preDestroy();
                }
                throw new Throwable( "unhealthy" + " = " + unhealthy.toString() );
            }
        }
        catch ( Throwable err ) {
            // ~ log only if healthy
            if ( isHealthy ) {
                logger.error( err.getMessage(), err );
            }

            //
            // ~ remove hook
            //
            boolean removed = Runtime.getRuntime().removeShutdownHook( shutdownHook );
            if ( removed ) {
                logger.info( "removed shutdown hook ..." );
            }

            //
            // ~ try to perform clean shutdown
            //
            if ( isHealthy ) {
                logger.error( "application failed to start, stopping lifecycle thread ..." );
                try {
                    shutdown();
                }
                catch ( Throwable t ) {
                    logger.warn( t.getMessage(), t );
                }
            }

            //
            // ~ we need to dump all threads for better troubleshooting
            //
            logger.info( "dumping all threads now before termination ... " );
            ThreadMXBean threadMXBean = ManagementFactory.getThreadMXBean();
            ThreadDump dump = new ThreadDump( threadMXBean );
            dump.dump( System.err );

            // send alert if application fails to start
            if ( isDevMode() ) {}
            else {
                try (Slf4jReporter reporter = Slf4jReporter.forRegistry( metricRegistry ).scheduleOn( platform ).build()) {
                    reporter.report();
                }
            }

            //
            // ~ re-throw error which will cause JVM exit with non-zero code
            //
            Throwables.throwIfUnchecked( err );
            throw new RuntimeException( err ); // docker(restart_policy=on-failure)
        }
        finally {
            rwLock.unlock();
        }
    }
    @Override
    public final void shutdown() throws Exception {
        Lock rwLock = lock.writeLock();
        rwLock.lock();
        try {
            status = ApplicationStatus.STOPPING;
            doStop();
        }
        finally {
            status = ApplicationStatus.STOPPED;
            rwLock.unlock();
        }
    }
    @Override
    public ApplicationStatus status() {
        Lock rLock = lock.readLock();
        rLock.lock();
        try {
            return status;
        }
        finally {
            rLock.unlock();
        }
    }
    @Override
    public DiEngine diEngine() {
        return Objects.requireNonNull( diEngine, "'bootstrap.diEngine' is not started yet" );
    }
    @Override
    public FixedSizePlatform platform() {
        return platform;
    }
    @Override
    public MetricRegistry metricRegistry() {
        return metricRegistry;
    }
    @Override
    public MeterRegistry meterRegisry() {
        return meterRegistry;
    }
    @Override
    public HealthCheckRegistry healthCheckRegistry() {
        return healthCheckRegistry;
    }
    @Override
    public Cloud cloud() {
        return cloud;
    }
    @Override
    public KeyStore keyStore() {
        return keyStore;
    }
    @Override
    public ApplicationProperties props() {
        return props;
    }
    @Override
    public String release() {
        return release;
    }
    @Override
    public boolean isDevMode() {
        return props.APP_DEV_MODE.get();
    }
    @Override
    public int port() {
        return props.CLOUD_APP_PORT.get();
    }
    @Override
    public String spaceName() {
        return props.CLOUD_APP_SPACE_NAME.get();
    }
    @Override
    public String appId() {
        return props.CLOUD_APP_ID.get();
    }
    @Override
    public boolean addChannel(Channel acceptor) {
        if ( status().isRunning() ) {
            return false;
        }
        return channels.add( acceptor );
    }
    @Override
    public boolean isHealthy() {
        SortedMap<String, HealthCheck.Result> results = healthCheckRegistry.runHealthChecks();
        return results.isEmpty();
    }
    @Override
    public void registerHealthCheck(String name, HealthCheck check) {
        healthCheckRegistry().register( name, check );
    }
    @Override
    public boolean addPlugin(BootstrapPlugin plugin) {
        if ( status().isRunning() ) {
            return false;
        }
        return plugins.add( plugin );
    }
    @Override
    public SentryClient sentry() {
        return sentry;
    }
    @Override
    public Tracer tracer() {
        return tracer;
    }
    @Override
    public Collection<BootstrapPlugin> plugins() {
        return Collections.unmodifiableCollection( plugins );
    }
    @Override
    public Collection<Channel> channels() {
        return Collections.unmodifiableCollection( channels );
    }
    @Override
    public Date startedAt() {
        return started;
    }
    @Override
    public boolean shouldSend(Event event) {
        return props.APP_SENTRY_ENABLED.get();
    }
    protected void doStart() throws Throwable {
        jmxReporter.start();

        for ( BootstrapPlugin plugin : plugins ) {
            if ( plugin instanceof BootstrapAware ) {
                ( (BootstrapAware) plugin ).setBootstrap( this );
            }
        }

        // ~ create parent injector for all plug-ins and register bindings
        parent = diEngine.configure( this );

        // ~ call post-construct action
        for ( BootstrapPlugin plugin : plugins ) {
            if ( plugin instanceof PostConstructable ) {
                logger.debug( "post-construct plugin '{}'", plugin );
                ( (PostConstructable) plugin ).postConstruct();
            }
        }

        // ~ initialize channels (without starting)
        for ( Channel acceptor : channels ) {
            parent.injectMembers( acceptor );
        }

        // ~ create DI module
        diEngine.up();

        // finally, accept in-bound traffic
        beforeAccept();
        for ( Channel acceptor : channels ) {
            acceptor.accept();
        }
    }
    protected void doStop() throws Exception {
        removeAllHealthChecks();

        // ~ first, stop all acceptors gracefully
        for ( Channel acceptor : channels ) {
            try {
                logger.info( "disposing acceptor {} ...", acceptor );
                acceptor.dispose();
            }
            catch ( Throwable t ) {
                logger.warn( t.getMessage(), t );
            }
        }
        try {
            if ( diEngine != null ) {
                logger.info( "closing application now ..." );
                diEngine.down();
            }
        }
        finally {
            // ~ shut down plug-ins
            for ( BootstrapPlugin plugin : plugins ) {
                if ( plugin instanceof PreDestroyable ) {
                    logger.debug( "pre-destroy plugin '{}'", plugin );
                    try {

                        ( (PreDestroyable) plugin ).preDestroy();
                    }
                    catch ( Throwable err ) {
                        logger.error( err.getMessage(), err );
                    }
                }
            }

            // ~ stop JMX reporter
            logger.info( "stoppping JMX reporter ..." );
            jmxReporter.close();

            // ~ shutdown executor
            logger.info( "shutting down platform now ..." );
            platform.preDestroy();

            // ~ close sentry
            if ( sentry != null ) {
                logger.info( "closing sentry now ..." );
                sentry.closeConnection();
            }

            // ~ cleanup action on configuration
            if ( props.APP_CLEAR_CFG_AT_SHUTDOWN_ENABLED.get() ) {
                logger.debug( "disposing CFG ..." );

                ApplicationConfig cfg = props.cfg();
                for ( String next : cfg.getConfigNames() ) {
                    Config removed = cfg.removeConfig( next );
                    if ( removed instanceof PollingDynamicConfig ) {
                        PollingDynamicConfig pdc = (PollingDynamicConfig) removed;
                        pdc.shutdown();
                    }
                }
                cfg.shutdown();
            }
        }
    }
    protected void beforeAccept() throws Exception {
        // NO-OP by default
    }
    private void removeAllHealthChecks() {
        for ( String name : healthCheckRegistry.getNames() ) {
            logger.info( "removing health-check {} ...", name );
            healthCheckRegistry.unregister( name );
        }
    }
}
