package com.turbospaces.boot;

import java.io.IOException;
import java.net.URI;
import java.util.Map.Entry;
import java.util.concurrent.CountDownLatch;

import javax.annotation.PreDestroy;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Provider;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.bootstrap.HttpServer;
import org.apache.http.impl.bootstrap.ServerBootstrap;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.http.protocol.HttpRequestHandlerMapper;
import org.apache.http.util.EntityUtils;
import org.junit.Assert;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.codahale.metrics.health.HealthCheck;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.Closeables;
import com.google.inject.Binder;
import com.google.inject.Binding;
import com.google.inject.Injector;
import com.google.inject.Key;
import com.google.inject.Module;
import com.google.inject.name.Names;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.common.ThrowableAction;
import com.turbospaces.di.AbstractBootstrapAwareModule;
import com.turbospaces.di.GuiceDiEngine;
import com.turbospaces.di.PreDestroyable;
import com.turbospaces.healthchecks.HttpHealthCheck;
import com.turbospaces.healthchecks.SocketHealthCheck;
import com.turbospaces.ups.PlainServiceInfo;

import io.jaegertracing.client.Version;

public class BootstrapTest {
    Logger logger = LoggerFactory.getLogger( getClass() );

    @Test
    public void works() throws Throwable {
        int port = PlatformUtil.freePort();

        MockCloud cfg = MockCloud.newMock();
        cfg.appPort( port );
        cfg.withKafka( 9092 );
        cfg.withZookeeper( 2181 );

        URIBuilder uri = new URIBuilder().setScheme( "http" ).setHost( "localhost" ).setPath( "/v1/status" ).setPort( port );

        ApplicationProperties props = new ApplicationProperties( cfg.build() );

        Bootstrap bootstrap = new SimpleBootstrap( props );
        bootstrap.addPlugin( new HttpClientPlugin() );
        bootstrap.addChannel( new InboundChannel( uri.build() ) );
        bootstrap.start( GuiceDiEngine.create( new DiModule() ) );

        try {
            Injector injector = (Injector) bootstrap.diEngine().injector();
            for ( Entry<Key<?>, Binding<?>> entry : injector.getAllBindings().entrySet() ) {
                logger.debug( entry.getValue().toString() );
            }

            injector.getInstance( CloseableHttpClient.class );
            Bean bean = injector.getInstance( Bean.class );
            logger.debug( bean.toString() );

            HealthCheck check1 = new HttpHealthCheck( props, uri.build() );
            Assert.assertTrue( check1.execute().isHealthy() );
            SocketHealthCheck check2 = new SocketHealthCheck( props, new PlainServiceInfo( "my-service-" + Version.get(), uri.toString() ) );
            Assert.assertTrue( check2.execute().isHealthy() );
        }
        finally {
            bootstrap.shutdown();
        }
    }

    public static class DiModule implements Module, BootstrapAware {
        private Bootstrap bootstrap;

        @Override
        public void setBootstrap(Bootstrap bootstrap) throws Exception {
            this.bootstrap = bootstrap;
        }
        @Override
        public void configure(Binder binder) {
            String appId = bootstrap.appId();
            Names.bindProperties( binder, ImmutableMap.of( "appId", appId ) );
            binder.bind( Bean.class ).asEagerSingleton();
        }
    }

    public static class InboundChannel implements Channel, BootstrapAware {
        private final Logger logger = LoggerFactory.getLogger( getClass() );
        private final URI uri;
        private ServerBootstrap http = ServerBootstrap.bootstrap();
        private Bootstrap bootstrap;
        private HttpServer channel;

        @Inject
        CloseableHttpClient httpClient;

        public InboundChannel(URI uri) {
            this.uri = uri;
        }
        @Override
        public void setBootstrap(Bootstrap bootstrap) throws Exception {
            this.bootstrap = bootstrap;
        }
        @Override
        public void accept() throws Exception {
            http = ServerBootstrap.bootstrap();
            http.setListenerPort( bootstrap.port() );
            http.setHandlerMapper( new HttpRequestHandlerMapper() {
                @Override
                public HttpRequestHandler lookup(HttpRequest request) {
                    return new HttpRequestHandler() {
                        @Override
                        public void handle(HttpRequest req, HttpResponse resp, HttpContext context) throws HttpException, IOException {
                            String reqUri = req.getRequestLine().getUri();
                            if ( reqUri.equals( uri.getPath() ) ) {
                                resp.setStatusCode( HttpStatus.SC_OK );
                                resp.setEntity( new StringEntity( req.toString() ) );
                            }
                            else {
                                resp.setStatusCode( HttpStatus.SC_NOT_FOUND );
                            }
                        }
                    };
                }
            } );
            channel = http.create();
            channel.start();

            HttpGet req = new HttpGet();
            req.setURI( uri );

            CountDownLatch latch = new CountDownLatch( 1 );
            try (CloseableHttpResponse resp = httpClient.execute( req )) {
                String text = EntityUtils.toString( resp.getEntity() );
                bootstrap.platform().work( new ThrowableAction() {
                    @Override
                    public void apply() throws Exception {
                        logger.debug( text );
                        latch.countDown();
                    }
                } );
            }
            latch.await();
        }
        @Override
        public void dispose() throws Exception {
            if ( channel != null ) {
                channel.stop();
            }
        }
    }

    public static class Bean implements BootstrapAware {
        private Bootstrap bootstrap;

        @Inject
        @Named("appId")
        String appId;

        @Inject
        CloseableHttpClient client;

        @Override
        public void setBootstrap(Bootstrap bootstrap) throws Exception {
            this.bootstrap = bootstrap;
        }
        @Override
        public String toString() {
            ToStringBuilder toString = PlatformUtil.noClassNameToString( this );
            return toString.append( "appId", appId ).append( "client", client ).append( "version", bootstrap.release() ).build();
        }
    }

    public static class HttpClientPlugin extends AbstractBootstrapAwareModule implements BootstrapPlugin, PreDestroyable, Provider<CloseableHttpClient> {
        private CloseableHttpClient client;

        @Override
        public CloseableHttpClient get() {
            if ( client == null ) {
                client = HttpClients.createMinimal();
            }
            return client;
        }
        @PreDestroy
        @Override
        public void preDestroy() throws Exception {
            Closeables.close( client, true );
        }
        @Override
        public void configure() {
            super.configure();
            bind( CloseableHttpClient.class ).toProvider( this ).asEagerSingleton();
        }
    }
}
