package org.springframework.cloud;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.UUID;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.openssl.PEMParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.app.ApplicationInstanceInfo;
import org.springframework.cloud.app.BasicApplicationInstanceInfo;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.common.CassandraServiceInfo;
import org.springframework.cloud.service.common.MysqlServiceInfo;
import org.springframework.cloud.service.common.OracleServiceInfo;
import org.springframework.cloud.service.common.PostgresqlServiceInfo;
import org.springframework.cloud.service.common.RedisServiceInfo;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.netflix.archaius.api.Config;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.common.SSL;
import com.turbospaces.ups.H2DatabaseServiceInfo;
import com.turbospaces.ups.HsqlDBServiceInfo;
import com.turbospaces.ups.MariaDBServiceInfo;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.RawServiceInfo;
import com.turbospaces.ups.ZookeeperServiceInfo;

//
// activated in non-CF standalone mode in docker/flink/VM (non-cf).
//
public class ConfigurableCloudConnector implements CloudConnector {
    public static final String FILE_SUFFIX = "_FILE";
    public static final String ENV_UPS_PREFIX = "UPS_";
    public static final String ENV_CACERT_PREFIX = "CACERT_";
    public static final String ENV_HOSTNAME = "HOSTNAME";
    public static final String ENV_SPACE_NAME = "SPACE_NAME";
    public static final String INDEX = "INDEX";
    public static final String RANDOM = "random";

    private final Logger logger = LoggerFactory.getLogger( getClass() );
    private final List<ServiceInfo> services;
    private final BasicApplicationInstanceInfo instanceInfo;

    private String spaceName;
    private String appId;
    private String instanceId;

    public ConfigurableCloudConnector(ApplicationProperties props, KeyStore keyStore) throws Exception {
        Map<String, ServiceInfo> l = new HashMap<>();
        Map<String, Object> cloudProps = new HashMap<>();

        configureInstance( props );
        configureAddress( props, cloudProps );
        configureSlot( props, cloudProps );
        configureSiteName( props, cloudProps );
        configureAppName( props, cloudProps );
        configureServices( props, keyStore, l );

        instanceInfo = new BasicApplicationInstanceInfo( instanceId, appId, cloudProps );
        services = ImmutableList.copyOf( l.values() );

        for ( ServiceInfo si : services ) {
            logger.debug( "adding ups('{}')={}", si.getId(), si );
        }
    }
    @Override
    public boolean isInMatchingCloud() {
        boolean isMatching = StringUtils.isNotEmpty( spaceName ) && StringUtils.isNotEmpty( appId );
        logger.info( "isMatching(spaceName={}, appId={}) = [{}]", spaceName, appId, isMatching );
        return isMatching;
    }
    @Override
    public ApplicationInstanceInfo getApplicationInstanceInfo() {
        return instanceInfo;
    }
    @Override
    public List<ServiceInfo> getServiceInfos() {
        return services;
    }
    private void configureAddress(ApplicationProperties props, Map<String, Object> cloudProps) throws IOException {
        ApplicationConfig cfg = props.cfg();
        String hostname = PlatformUtil.detectIp();
        int port = ApplicationProperties.DEFAULT_PORT;

        if ( cfg.containsKey( CloudOptions.CLOUD_APP_HOST ) ) {
            hostname = cfg.getString( CloudOptions.CLOUD_APP_HOST );
        }
        if ( cfg.containsKey( CloudOptions.CLOUD_APP_PORT ) ) {
            // ~ random port
            if ( RANDOM.equalsIgnoreCase( cfg.getString( CloudOptions.CLOUD_APP_PORT ) ) ) {
                port = PlatformUtil.freePort();
            }
            else {
                port = cfg.getInteger( CloudOptions.CLOUD_APP_PORT );
            }
        }

        // k8s or docker SWARM
        if ( StringUtils.isNotEmpty( System.getenv( ENV_HOSTNAME ) ) ) {
            hostname = System.getenv( ENV_HOSTNAME );
        }

        addCloudProp( CloudOptions.CLOUD_APP_HOST, hostname, cloudProps );
        addCloudProp( CloudOptions.CLOUD_APP_PORT, port, cloudProps );
    }
    private void configureSlot(ApplicationProperties props, Map<String, Object> cloudProps) {
        ApplicationConfig cfg = props.cfg();
        String slot = null;

        if ( cfg.containsKey( CloudOptions.CLOUD_APP_INSTANCE_INDEX ) ) {
            slot = cfg.getString( CloudOptions.CLOUD_APP_INSTANCE_INDEX );
            if ( RANDOM.equalsIgnoreCase( slot ) ) {
                slot = RandomStringUtils.randomAlphanumeric( getClass().getSimpleName().length() );
            }
        }

        // k8s or docker SWARM
        if ( StringUtils.isNotEmpty( System.getenv( INDEX ) ) ) {
            slot = System.getenv( INDEX );
        }

        if ( StringUtils.isEmpty( slot ) ) {
            slot = String.valueOf( 0 );
        }

        addCloudProp( CloudOptions.CLOUD_APP_INSTANCE_INDEX, slot, cloudProps );
    }
    private void configureSiteName(ApplicationProperties props, Map<String, Object> cloudProps) {
        ApplicationConfig cfg = props.cfg();
        spaceName = System.getenv( ENV_SPACE_NAME );

        if ( cfg.containsKey( CloudOptions.CLOUD_APP_SPACE_NAME ) ) {
            spaceName = cfg.getString( CloudOptions.CLOUD_APP_SPACE_NAME );
        }
        if ( StringUtils.isNotEmpty( spaceName ) ) {
            addCloudProp( CloudOptions.CLOUD_APP_SPACE_NAME, spaceName, cloudProps );
        }
    }
    private void configureInstance(ApplicationProperties props) {
        ApplicationConfig cfg = props.cfg();
        if ( cfg.containsKey( CloudOptions.CLOUD_APP_ID ) ) {
            appId = cfg.getString( CloudOptions.CLOUD_APP_ID );
        }
        instanceId = UUID.randomUUID().toString();
        if ( cfg.containsKey( CloudOptions.CLOUD_APP_INSTANCE_ID ) ) {
            instanceId = cfg.getString( CloudOptions.CLOUD_APP_INSTANCE_ID );
        }
    }
    private void configureAppName(ApplicationProperties props, Map<String, Object> cloudProps) {
        ApplicationConfig cfg = props.cfg();
        if ( cfg.containsKey( CloudOptions.CLOUD_APP_NAME ) ) {
            String appName = cfg.getString( CloudOptions.CLOUD_APP_NAME );
            addCloudProp( CloudOptions.CLOUD_APP_NAME, appName, cloudProps );
        }
    }
    private void configureServices(ApplicationProperties props, KeyStore keyStore, Map<String, ServiceInfo> map) throws Exception {
        ApplicationConfig cfg = props.cfg();
        Config prefixedView = cfg.getPrefixedView( "service" );
        Iterator<String> it = prefixedView.getKeys();
        while ( it.hasNext() ) {
            String key = it.next();
            int idx = key.indexOf( ".uri" );
            if ( idx > 0 ) {
                String serviceId = key.substring( 0, idx );
                String value = prefixedView.getRawProperty( key ).toString();
                RawServiceInfo raw = new RawServiceInfo( serviceId, value.getBytes( StandardCharsets.UTF_8 ) );
                map.put( serviceId, raw );

                try {
                    URI serviceUri = new URI( value );
                    if ( StringUtils.isNotEmpty( serviceUri.getScheme() ) ) {
                        ServiceInfo serviceInfo = toServiceInfo( serviceId, serviceUri );
                        map.put( serviceId, serviceInfo );
                    }
                }
                catch ( URISyntaxException err ) {

                }
            }
        }

        Collection<Certificate> cacerts = Lists.newLinkedList();
        for ( Entry<String, String> entry : System.getenv().entrySet() ) {
            if ( entry.getKey().startsWith( ENV_CACERT_PREFIX ) ) {
                File pem = new File( entry.getValue() ); // ~ PEM
                byte[] bytes = Files.readAllBytes( pem.toPath() );
                String raw = new String( bytes, StandardCharsets.UTF_8 );
                try (StringReader reader = new StringReader( raw )) {
                    try (PEMParser parser = new PEMParser( reader )) {
                        org.bouncycastle.cert.X509CertificateHolder keyInfo = (org.bouncycastle.cert.X509CertificateHolder) parser.readObject();

                        try (ByteArrayInputStream io = new ByteArrayInputStream( keyInfo.getEncoded() )) {
                            CertificateFactory fact = CertificateFactory.getInstance( "X.509" );
                            X509Certificate cer = (X509Certificate) fact.generateCertificate( io );
                            cacerts.add( cer );
                        }
                    }
                }
            }
            else if ( entry.getKey().startsWith( ENV_UPS_PREFIX ) ) {
                String key = entry.getKey();
                String value = entry.getValue();
                String serviceId = key.substring( ENV_UPS_PREFIX.length() ).trim().toLowerCase();

                // docker mounts secrets to /run/secrets/${secret}
                if ( key.endsWith( FILE_SUFFIX ) ) {
                    File f = new File( value );
                    byte[] encoded = Files.readAllBytes( f.toPath() );
                    value = new String( encoded, StandardCharsets.UTF_8 );
                    serviceId = serviceId.substring( 0, serviceId.length() - FILE_SUFFIX.length() );
                    logger.debug( "secret={} has been loaded from={} ...", serviceId, f );
                }

                RawServiceInfo raw = new RawServiceInfo( serviceId, value.getBytes( StandardCharsets.UTF_8 ) );
                map.put( serviceId, raw );

                try {
                    URI serviceUri = new URI( value );
                    if ( StringUtils.isNotEmpty( serviceUri.getScheme() ) ) {
                        ServiceInfo serviceInfo = toServiceInfo( serviceId, serviceUri );
                        map.put( serviceId, serviceInfo );
                    }
                }
                catch ( URISyntaxException err ) {

                }
            }
        }

        if ( cacerts.isEmpty() ) {}
        else {
            SSL.addCertificates( keyStore, cacerts );
        }
    }
    private static ServiceInfo toServiceInfo(String id, URI uri) {
        String scheme = uri.getScheme();
        String[] details = scheme.split( "\\+" );
        String protocol = details[details.length - 1];

        switch ( protocol ) {
            case "oracle": {
                return new OracleServiceInfo( id, uri.toString() );
            }
            case "postgres": {
                return new PostgresqlServiceInfo( id, uri.toString() );
            }
            case "mysql": {
                return new MysqlServiceInfo( id, uri.toString() );
            }
            case "mariadb": {
                return new MariaDBServiceInfo( id, uri.toString() );
            }
            case "hsqldb": {
                return new HsqlDBServiceInfo( id, uri.toString() );
            }
            case "h2": {
                return new H2DatabaseServiceInfo( id, uri.toString() );
            }
            case "redis":
            case "rediss": {
                return new RedisServiceInfo( id, uri.toString() );
            }
            case "kafka": {
                return new PlainServiceInfo( id, uri.toString() );
            }
            case "zk": {
                return new ZookeeperServiceInfo( id, uri );
            }
            case "cassandra": {
                Entry<List<String>, Integer> entry = PlatformUtil.parseNodesAndPort( uri );
                return new CassandraServiceInfo( id, entry.getKey(), entry.getValue() );
            }
            case "http":
            case "https":
            case "tcp":
            case "file": {
                return new PlainServiceInfo( id, uri.toString() );
            }
            default: {
                throw new IllegalArgumentException( "unknown schema: " + protocol );
            }
        }
    }
    private void addCloudProp(String key, Object value, Map<String, Object> props) {
        logger.debug( "adding {}={}", key, value );
        props.put( key, Objects.requireNonNull( value ) );
    }
}
