package com.turbospaces.ups;

import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.cloud.Cloud;
import org.springframework.cloud.CloudConnector;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.UriBasedServiceInfo;
import org.springframework.cloud.service.common.CassandraServiceInfo;

import com.google.common.net.HostAndPort;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.cfg.ApplicationConfig;

public interface UPSs {
    String CFG = "cfg";
    //
    String ZOOKEEPER = "zookeeper";
    String KAFKA = "kafka";
    String JAEGER = "jaeger";
    String SENTRY = "sentry";
    String ELASTIC_SEARCH = "elastic-search";

    //
    String REDIS = "redis";
    String CASSANDRA = "cassandra";
    //
    String GRAPHITE = "graphite";
    String INFLUX_DB = "influx-db";
    //
    String WEB3J = "web3j";
    String WEB3J_CREDENTIALS = "web3j-credentials";
    //
    String MYSQL_OWNER = "mysql-owner";
    String MYSQL_APP = "mysql-app";
    String ORACLE_OWNER = "oracle-owner";
    String ORACLE_APP = "oracle-app";
    String POSTGRES_OWNER = "postgres-owner";
    String POSTGRES_APP = "postgres-app";
    String QUARTZ_OWNER = "quartz-owner";
    String QUARTZ_APP = "quartz-app";
    String H2_OWNER = "h2-owner";
    String H2_APP = "h2-app";
    //
    String JOB_MANAGER = "job-manager";
    String TASK_MANAGER = "task-manager";
    //
    String FRONTEND_SERVER = "frontend-server";
    String SERVER = "server";
    String GATEWAY = "gateway";
    String FRONTEND_UI = "frontend-ui";
    String PUSH_SERVER = "push-server";
    //
    String SENDGRID = "sendgrid";
    String MAILCHIMP = "mailchimp";

    String WORLDPAY = "worldpay";
    String PAYPAL = "paypal";

    String GOOGLE_PAY = "google-pay";
    String GOOGLE_PLAY = "google-play";
    String GOOGLE_ADMOB = "google-admob";
    String GOOGLE_VERIFIER = "google-verifier";

    String FACEBOOK = "facebook";
    String INSTAGRAM = "instagram";

    String SLACK = "slack";

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> Optional<T> findServiceInfoByType(Cloud cloud, Class<T> serviceInfoClass) {
        List<ServiceInfo> serviceInfos = cloud.getServiceInfos();
        List<ServiceInfo> l = serviceInfos.stream().filter( new Predicate<ServiceInfo>() {
            @Override
            public boolean test(ServiceInfo input) {
                return input.getClass().equals( serviceInfoClass );
            }
        } ).collect( Collectors.toList() );
        Iterator<ServiceInfo> it = l.iterator();
        if ( it.hasNext() ) {
            T next = (T) it.next();
            if ( it.hasNext() ) {
                throw new IllegalArgumentException( "unable to find unique service by type" );
            }
            return Optional.of( next );
        }
        return Optional.empty();
    }
    static <T extends ServiceInfo> T findRequiredServiceInfoByType(Bootstrap boot, Class<T> serviceInfoClass) {
        return findServiceInfoByType( boot.cloud(), serviceInfoClass ).get();
    }
    static <T extends ServiceInfo> T findRequiredServiceInfoByType(Cloud cloud, Class<T> serviceInfoClass) {
        return findServiceInfoByType( cloud, serviceInfoClass ).get();
    }
    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> Optional<T> findServiceInfoByName(List<ServiceInfo> serviceInfos, String name) {
        return (Optional<T>) serviceInfos.stream().filter( new Predicate<ServiceInfo>() {
            @Override
            public boolean test(ServiceInfo input) {
                return input.getId().equals( name );
            }
        } ).findAny();
    }
    static <T extends ServiceInfo> Optional<T> findScopedServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<T> opt = findServiceInfoByName( bootstrap, scope + "-" + name );
        if ( opt.isPresent() ) {
            return opt;
        }
        opt = findServiceInfoByName( bootstrap.cloud(), name );
        return opt;
    }
    static <T extends ServiceInfo> Optional<T> findServiceInfoByName(Bootstrap boot, String name) {
        List<ServiceInfo> serviceInfos = boot.cloud().getServiceInfos();
        return findServiceInfoByName( serviceInfos, name );
    }
    static <T extends ServiceInfo> Optional<T> findServiceInfoByName(Cloud cloud, String name) {
        List<ServiceInfo> serviceInfos = cloud.getServiceInfos();
        return findServiceInfoByName( serviceInfos, name );
    }
    static <T extends ServiceInfo> Optional<T> findServiceInfoByName(CloudConnector connector, String name) {
        List<ServiceInfo> serviceInfos = connector.getServiceInfos();
        return findServiceInfoByName( serviceInfos, name );
    }
    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findScopedRequiredServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<ServiceInfo> opt = findServiceInfoByName( bootstrap, scope + "-" + name );
        if ( opt.isPresent() ) {
            return (T) opt.get();
        }
        opt = findServiceInfoByName( bootstrap.cloud(), name );
        return (T) opt.get();
    }
    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findRequiredServiceInfoByName(Bootstrap bootstrap, String name) {
        return (T) findServiceInfoByName( bootstrap.cloud(), name ).get();
    }
    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findRequiredServiceInfoByName(Cloud cloud, String name) {
        return (T) findServiceInfoByName( cloud, name ).get();
    }
    static ApplicationConfig addUserProvideServices(ApplicationConfig cfg, List<ServiceInfo> services) {
        Properties props = new Properties();
        addUserProvideServices( props, services );
        cfg.setLocalProperties( props );
        return cfg;
    }
    static Properties addUserProvideServices(Properties props, List<ServiceInfo> services) {
        for ( ServiceInfo info : services ) {
            String uri = null;
            if ( info instanceof UriBasedServiceInfo ) {
                uri = ( (UriBasedServiceInfo) info ).getUri();
            }
            else if ( info instanceof ZookeeperServiceInfo ) {
                uri = ( (ZookeeperServiceInfo) info ).toString();
            }
            else if ( info instanceof CassandraServiceInfo ) {
                uri = ( (CassandraServiceInfo) info ).toString();
            }
            else if ( info instanceof RawServiceInfo ) {
                uri = new String( ( (RawServiceInfo) info ).getPayload(), StandardCharsets.UTF_8 );
            }
            else {
                throw new IllegalArgumentException();
            }

            props.put( "service." + info.getId() + ".uri", uri );
        }
        return props;
    }
    static PlainServiceInfo toKafkaServiceInfo(HostAndPort addr) {
        for ( ;; ) {
            try {
                String uri = String.format( "kafka://%s:%d", addr.getHost(), addr.getPort() );
                PlainServiceInfo info = new PlainServiceInfo( UPSs.KAFKA, uri );
                return info;
            }
            catch ( Exception err ) {
                ExceptionUtils.wrapAndThrow( err );
            }
        }
    }
    static ZookeeperServiceInfo toZookeeperServiceInfo(HostAndPort addr) {
        for ( ;; ) {
            try {
                URI uri = new URI( String.format( "zk://%s:%d", addr.getHost(), addr.getPort() ) );
                ZookeeperServiceInfo info = new ZookeeperServiceInfo( UPSs.ZOOKEEPER, uri );
                return info;
            }
            catch ( Exception err ) {
                ExceptionUtils.wrapAndThrow( err );
            }
        }
    }
    static PlainServiceInfo toWeb3jServiceInfo(HostAndPort addr) {
        PlainServiceInfo info = new PlainServiceInfo( UPSs.WEB3J, String.format( "http://%s:%s", addr.getHost(), addr.getPort() ) );
        return info;
    }
    static CassandraServiceInfo toCassandraServiceInfo(HostAndPort addr) {
        List<String> peer = Collections.singletonList( addr.getHost() );
        CassandraServiceInfo info = new CassandraServiceInfo( UPSs.CASSANDRA, peer, addr.getPort() ) {
            @Override
            public String toString() {
                return String.format( "%s://%s:%s", "cassandra", StringUtils.join( peer, ',' ), getPort() );
            }
        };
        return info;
    }
}
