package com.turbospaces.ups;

import java.util.Arrays;
import java.util.List;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.springframework.cloud.service.BaseServiceInfo;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.UriBasedServiceInfo;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.common.EnvUtil;

import io.netty.handler.codec.http.QueryStringDecoder;
import reactor.core.publisher.Flux;

public interface UPSs {
    String CFG = "cfg";
    String SENTRY = "sentry";
    String ELASTIC_SEARCH = "elastic-search";
    String INFLUX = "influx";
    String JAEGER = "jaeger";
    String TEMPORAL = "temporal";
    String MAXMIND = "maxmind";
    String CONFIG_CAT = "config-cat";
    String TURNSTILE = "turnstile";
    String RECAPTCHA = "recaptcha";
    String REDIS = "redis";
    String DOCUMENTATION = "documentation";

    //
    // ~ multiple peers potentially
    //
    String KAFKA = "kafka";

    String QUARTZ_APP = "quartz-app";
    String UPS_GCV = "google-application-credentials";

    //
    // ~ PG
    //
    String POSTGRES_OWNER = "postgres-owner";
    String POSTGRES_APP = "postgres-app";

    //
    // ~ testing
    //
    String H2_OWNER = "h2-owner";
    String H2_APP = "h2-app";
    String PUBSUB = "pubsub";

    <T extends ServiceInfo> Flux<ServiceInfo> serviceInfoByName(String ups);

    <T extends ServiceInfo> Flux<ServiceInfo> scopedServiceInfoByName(String scope, String name);

    static <T extends ServiceInfo> Optional<T> findServiceInfoByName(Bootstrap boot, String name) {
        var env = EnvUtil.envStage();

        if (env.isPresent()) {
            var scoped = String.format("%s-%s", env.get(), name);

            return UPSs.<T> findServiceInfo(boot, scoped).or(() -> findServiceInfo(boot, name));
        }

        return findServiceInfo(boot, name);
    }

    static <T extends ServiceInfo> Optional<T> findScopedServiceInfoByName(Bootstrap bootstrap, String name, String... scopes) {
        // ~ skip nulls in the end
        if (scopes.length > 0 && StringUtils.isEmpty(scopes[scopes.length - 1])) {
            return findScopedServiceInfoByName(bootstrap, name, Arrays.copyOfRange(scopes, 0, scopes.length - 1));
        }
        for (int i = scopes.length; i > 0; i--) {
            String joined = Joiner.on("-").join(Arrays.copyOfRange(scopes, 0, i));
            Optional<T> serviceMaybe = findServiceInfoByName(bootstrap, String.format("%s-%s", joined.toLowerCase(), name));
            if (serviceMaybe.isPresent()) {
                return serviceMaybe;
            }
        }
        return findServiceInfoByName(bootstrap, name);
    }

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findRequiredScopedServiceInfoByName(Bootstrap bootstrap, String name, String... scopes) {
        return (T) findScopedServiceInfoByName(bootstrap, name, scopes).get();
    }

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findRequiredServiceInfoByName(Bootstrap bootstrap, String name) {
        return (T) findServiceInfoByName(bootstrap, name).get();
    }

    static <T extends ServiceInfo> Optional<T> findScopedServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<T> opt = findServiceInfoByName(bootstrap, String.format("%s-%s", scope, name));
        if (opt.isPresent()) {
            return opt;
        }
        opt = findServiceInfoByName(bootstrap, name);
        return opt;
    }

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findScopedRequiredServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<ServiceInfo> opt = findServiceInfoByName(bootstrap, String.format("%s-%s", scope, name));
        if (opt.isPresent()) {
            return (T) opt.get();
        }
        opt = findServiceInfoByName(bootstrap, name);
        return (T) opt.get();
    }

    @SuppressWarnings("unchecked")
    private static <T extends ServiceInfo> Optional<T> findServiceInfo(Bootstrap boot, String name) {
        List<ServiceInfo> serviceInfos = boot.cloud().getServiceInfos();

        return (Optional<T>) serviceInfos.stream().filter(input -> input.getId().equals(name)).findAny();
    }

    static boolean isEquals(ServiceInfo t, ServiceInfo u) {
        if (t.getClass().equals(u.getClass())) {
            if (t instanceof UriBasedServiceInfo turi && u instanceof UriBasedServiceInfo uuri) {
                return new EqualsBuilder().append(turi.getId(), uuri.getId()).append(turi.getUri(), uuri.getUri()).isEquals();
            }
            if (t instanceof RawServiceInfo turi && u instanceof RawServiceInfo uuri) {
                return new EqualsBuilder().append(turi.getId(), uuri.getId()).append(turi.getPayload(), uuri.getPayload()).isEquals();
            }
            if (t instanceof BaseServiceInfo turi && u instanceof BaseServiceInfo uuri) {
                return Objects.equals(turi.getId(), uuri.getId());
            }

            throw new UnsupportedOperationException("don't know how to compare " + t.getClass().getSimpleName());
        }
        return false;
    }

    static int hashCode(ServiceInfo si) {
        if (si instanceof UriBasedServiceInfo usi) {
            return Objects.hash(usi.getId(), usi.getUri());
        }
        if (si instanceof RawServiceInfo rsi) {
            return Objects.hash(rsi.getId(), Arrays.hashCode(rsi.getPayload()));
        }
        if (si instanceof BaseServiceInfo bsi) {
            return Objects.hash(bsi.getId());
        }
        throw new UnsupportedOperationException("don't know how to calculate hashcode " + si.getClass().getSimpleName());
    }

    static Optional<String> getQueryParam(String key, PlainServiceInfo info) {
        if (StringUtils.isNotEmpty(info.getQuery())) {
            QueryStringDecoder decoder = new QueryStringDecoder(info.getQuery(), false);
            for (Entry<String, List<String>> entry : decoder.parameters().entrySet()) {
                if (entry.getKey().equals(key)) {
                    List<String> list = entry.getValue();
                    String toReturn = Iterables.getOnlyElement(list);
                    return Optional.of(toReturn);
                }
            }
        }
        return Optional.empty();
    }

    static String getRequiredQueryParam(String key, PlainServiceInfo info) throws Exception {
        Optional<String> opt = getQueryParam(key, info);
        Preconditions.checkArgument(opt.isPresent(), "param '%s' is not found or miss-configured", key);
        return opt.get();
    }
}
