package com.turbospaces.executor;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;

import org.slf4j.MDC;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;

import io.netty.util.AsciiString;
import io.vavr.CheckedFunction0;
import io.vavr.CheckedRunnable;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class SerialContextWorker implements ContextWorker {
    private final AsciiString key;
    private final Executor executor;

    public SerialContextWorker(AsciiString key, ExecutorService executor) {
        this.key = Objects.requireNonNull(key);
        this.executor = MoreExecutors.newSequentialExecutor(executor);
    }
    @Override
    public ContextWorker forKey(WorkUnit unit) {
        throw new UnsupportedOperationException();
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    public ListenableFuture<?> submit(CheckedRunnable command) {
        long now = System.currentTimeMillis();
        Map<String, String> mdc = MDC.getCopyOfContextMap(); // ~ capture MDC
        SettableFuture<Object> settable = SettableFuture.create();

        try {
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    Thread currentThread = Thread.currentThread();
                    String oldName = currentThread.getName();
                    String newName = oldName + "|" + key;
                    currentThread.setName(newName);

                    PlatformExecutorService.propagete(mdc);

                    try {
                        long delta = System.currentTimeMillis() - now;
                        if (delta > 0) {
                            log.debug("task({}) was queued for {} ms before exec for key: {}", Math.abs(hashCode()), delta, key);
                        }

                        command.run();
                        settable.set(new Object());
                    } catch (Throwable err) {
                        settable.setException(err);
                    } finally {
                        currentThread.setName(oldName);

                        MDC.clear();
                    }
                }
            });
        } catch (RejectedExecutionException err) {
            settable.setException(err);
        }

        return settable;
    }
    @Override
    public <T> ListenableFuture<T> submit(CheckedFunction0<T> command) {
        long now = System.currentTimeMillis();
        Map<String, String> mdc = MDC.getCopyOfContextMap(); // ~ capture MDC
        SettableFuture<T> settable = SettableFuture.create();

        try {
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    Thread currentThread = Thread.currentThread();
                    String oldName = currentThread.getName();
                    String newName = oldName + "|" + key;
                    currentThread.setName(newName);

                    PlatformExecutorService.propagete(mdc);

                    try {
                        long delta = System.currentTimeMillis() - now;
                        if (delta > 0) {
                            log.debug("task({}) was queued for {} ms before exec for key: {}", Math.abs(hashCode()), delta, key);
                        }

                        settable.set(command.apply());
                    } catch (Throwable err) {
                        settable.setException(err);
                    } finally {
                        currentThread.setName(oldName);

                        MDC.clear();
                    }
                }
            });
        } catch (RejectedExecutionException err) {
            settable.setException(err);
        }

        return settable;
    }
}
