package com.turbospaces.statemachine;

import java.util.Random;
import java.util.UUID;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.statemachine.ExtendedState;
import org.springframework.statemachine.StateContext;
import org.springframework.statemachine.StateMachine;
import org.springframework.statemachine.action.Action;
import org.springframework.statemachine.config.EnableStateMachineFactory;
import org.springframework.statemachine.config.EnumStateMachineConfigurerAdapter;
import org.springframework.statemachine.config.StateMachineFactory;
import org.springframework.statemachine.config.builders.StateMachineStateConfigurer;
import org.springframework.statemachine.config.builders.StateMachineTransitionConfigurer;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestExecutionListeners;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import lombok.extern.slf4j.Slf4j;
import reactor.core.publisher.Mono;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = com.turbospaces.statemachine.StateMachineTest.AppConfig.class)
@TestExecutionListeners(mergeMode = TestExecutionListeners.MergeMode.MERGE_WITH_DEFAULTS, listeners = {})
public class StateMachineTest {
    @Autowired
    private StateMachineFactory<KycVerificationStates, KycStatusEvents> stateMachineFactory;

    @Test
    public void works() {
        StateMachine<KycVerificationStates, KycStatusEvents> m1 = stateMachineFactory.getStateMachine("demo");
        m1.startReactively().subscribe();
        m1.sendEvent(
                Mono.just(MessageBuilder.withPayload(KycStatusEvents.AFTER_SOME_INITIAL_EVENT)
                        .setHeader("header-1", UUID.randomUUID().toString())
                        .setHeader("header-2", UUID.randomUUID().toString())
                        .build()))
                .blockFirst();
        Assertions.assertEquals(KycVerificationStates.PUBLIC_PAGE_REVIEW, m1.getState().getId());
    }

    @Configuration
    @Slf4j
    @EnableStateMachineFactory
    public static class AppConfig extends EnumStateMachineConfigurerAdapter<KycVerificationStates, KycStatusEvents> {
        @Override
        public void configure(StateMachineStateConfigurer<KycVerificationStates, KycStatusEvents> states) throws Exception {
            states.withStates()
                    .initial(KycVerificationStates.INITIAL)
                    .state(KycVerificationStates.PUBLIC_PAGE_REVIEW)
                    .state(KycVerificationStates.DOC_REVIEW)
                    .end(KycVerificationStates.CONFIRMED)
                    .end(KycVerificationStates.DECLINED);
        }
        @Override
        public void configure(StateMachineTransitionConfigurer<KycVerificationStates, KycStatusEvents> transitions) throws Exception {
            transitions.withExternal()
                    .source(KycVerificationStates.INITIAL)
                    .target(KycVerificationStates.PUBLIC_PAGE_REVIEW)
                    .event(KycStatusEvents.AFTER_SOME_INITIAL_EVENT)
                    .action(new Action<KycVerificationStates, KycStatusEvents>() {
                        @Override
                        public void execute(StateContext<KycVerificationStates, KycStatusEvents> context) {
                            KycVerificationStates source = context.getSource().getId();
                            KycVerificationStates target = context.getTarget().getId();
                            ExtendedState extendedState = context.getExtendedState();

                            log.info("applying an action source: {}, target: {}", source, target);
                            extendedState.getVariables().put("counter", new Random().nextInt());
                        }
                    });
        }
    }
}
