package com.turbospaces.cfg;

import com.netflix.archaius.api.Property;
import com.netflix.archaius.api.exceptions.ConfigException;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;

public class DynamicPropertyFactoryTest {

    private static final String PROPERTY_NAME = "soft-kyc-required-before-purchase.percentage.users-applied";
    private static final int DEFAULT_VALUE = 1;
    private static final String WRONG_VALUE = "50akDFs83$#%#";
    private static final int CORRECT_VALUE = 50;

    @Test
    public void testOrElseGetCorrectValue() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, CORRECT_VALUE);

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.orElse(DEFAULT_VALUE).get();

        assertEquals(CORRECT_VALUE, integerValue);
    }

    @Test
    public void testOrElseGetWrongValueDefault() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, WRONG_VALUE);

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.orElse(DEFAULT_VALUE).get();

        assertEquals(DEFAULT_VALUE, integerValue);
    }

    @Test
    public void testOrElseGetNullValueDefault() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, null);

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.orElse(DEFAULT_VALUE).get();

        assertEquals(DEFAULT_VALUE, integerValue);
    }

    @Test
    public void testOrElseGetEmptyValueDefault() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, "");

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.orElse(DEFAULT_VALUE).get();

        assertEquals(DEFAULT_VALUE, integerValue);
    }

    @Test
    public void testGetCorrectValue() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, CORRECT_VALUE);

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.orElse(DEFAULT_VALUE).get();

        assertEquals(CORRECT_VALUE, integerValue);
    }

    @Test
    public void testWrongValueDefault() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, WRONG_VALUE);

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.orElse(DEFAULT_VALUE).get();

        assertEquals(DEFAULT_VALUE, integerValue);
    }

    @Test
    public void testGetNullValueNull() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, null);

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.get();

        assertNull(integerValue);
    }

    @Test
    public void testGetEmptyValueNull() throws ConfigException {
        DynamicPropertyFactory factory = createDynamicPropertyFactory(PROPERTY_NAME, "");

        Property<Integer> integerProperty = factory.get(PROPERTY_NAME, Integer.class);
        Integer integerValue = integerProperty.get();

        assertNull(integerValue);
    }

    private static DynamicPropertyFactory createDynamicPropertyFactory(String name, String value) throws ConfigException {
        ApplicationConfig config = ApplicationConfig.create();
        config.setLocalProperty(name, value);
        return DynamicPropertyFactory.from(config);
    }


    private static DynamicPropertyFactory createDynamicPropertyFactory(String name, int value) throws ConfigException {
        ApplicationConfig config = ApplicationConfig.create();
        config.setLocalProperty(name, String.valueOf(value));
        return DynamicPropertyFactory.from(config);
    }
}