package com.turbospaces.executor;

import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import org.springframework.beans.factory.BeanNameAware;

import com.google.common.cache.LoadingCache;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.Bootstrap;

public abstract class AbstractContextWorkerFactory<T, U extends WorkUnit> extends AbstractBootstrapAware implements ContextWorkerFactory<U>, BeanNameAware {
    protected LoadingCache<T, ThreadPoolContextWorker> workers;
    private String beanName;

    @Override
    public void setBeanName(String name) {
        this.beanName = Objects.requireNonNull(name);
    }
    @Override
    public void setBootstrap(Bootstrap bootstrap) {
        super.setBootstrap(bootstrap);
        if (Objects.nonNull(workers)) {
            Duration duration = bootstrap.props().APP_TIMER_INTERVAL.get();
            Integer maxSize = bootstrap.props().CACHE_DEFAULT_MAX_SIZE.get();
            bootstrap.globalPlatform().scheduleWithFixedDelay(new Runnable() {
                @Override
                public void run() {
                    workers.cleanUp();
                    //
                    // ~ we want to raise sentry alert pro-actively and not to wait for OOM
                    //
                    if (workers.size() > maxSize) {
                        logger.error("cache : ({}) exceeded reasonable size: {}, current: {}", beanName, maxSize, workers.size());
                    }
                }
            }, 0, duration.toSeconds(), TimeUnit.SECONDS);
        }
    }
}
