package com.turbospaces.ups;

import java.util.Arrays;
import java.util.List;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.springframework.cloud.service.BaseServiceInfo;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.UriBasedServiceInfo;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.turbospaces.boot.Bootstrap;

import io.netty.handler.codec.http.QueryStringDecoder;
import reactor.core.publisher.Flux;

public interface UPSs {
    String CFG = "cfg";
    String SENTRY = "sentry";
    String ELASTIC_SEARCH = "elastic-search";
    String INFLUX = "influx";
    String JAEGER = "jaeger";
    String TEMPORAL = "temporal";
    String MAXMIND = "maxmind";
    String CONFIG_CAT = "config-cat";
    String TURNSTILE = "turnstile";
    String REDIS = "redis";

    //
    // ~ multiple peers potentially
    //
    String KAFKA = "kafka";

    String QUARTZ_APP = "quartz-app";
    String UPS_GCV = "google-application-credentials";

    //
    // ~ PG
    //
    String POSTGRES_OWNER = "postgres-owner";
    String POSTGRES_APP = "postgres-app";

    //
    // ~ testing
    //
    String H2_OWNER = "h2-owner";
    String H2_APP = "h2-app";
    String PUBSUB = "pubsub";

    <T extends ServiceInfo> Flux<ServiceInfo> serviceInfoByName(String ups);

    <T extends ServiceInfo> Flux<ServiceInfo> scopedServiceInfoByName(String scope, String name);

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> Optional<T> findServiceInfoByName(Bootstrap boot, String name) {
        List<ServiceInfo> serviceInfos = boot.cloud().getServiceInfos();
        return (Optional<T>) serviceInfos.stream().filter(new Predicate<ServiceInfo>() {
            @Override
            public boolean test(ServiceInfo input) {
                return input.getId().equals(name);
            }
        }).findAny();
    }

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findRequiredServiceInfoByName(Bootstrap bootstrap, String name) {
        return (T) findServiceInfoByName(bootstrap, name).get();
    }

    static <T extends ServiceInfo> Optional<T> findScopedServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<T> opt = findServiceInfoByName(bootstrap, String.format("%s-%s", scope, name));
        if (opt.isPresent()) {
            return opt;
        }
        opt = findServiceInfoByName(bootstrap, name);
        return opt;
    }

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findScopedRequiredServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<ServiceInfo> opt = findServiceInfoByName(bootstrap, String.format("%s-%s", scope, name));
        if (opt.isPresent()) {
            return (T) opt.get();
        }
        opt = findServiceInfoByName(bootstrap, name);
        return (T) opt.get();
    }

    static boolean isEquals(ServiceInfo t, ServiceInfo u) {
        if (t.getClass().equals(u.getClass())) {
            if (t instanceof UriBasedServiceInfo && u instanceof UriBasedServiceInfo) {
                UriBasedServiceInfo turi = (UriBasedServiceInfo) t;
                UriBasedServiceInfo uuri = (UriBasedServiceInfo) u;

                return new EqualsBuilder().append(turi.getId(), uuri.getId()).append(turi.getUri(), uuri.getUri()).isEquals();
            }
            if (t instanceof RawServiceInfo && u instanceof RawServiceInfo) {
                RawServiceInfo turi = (RawServiceInfo) t;
                RawServiceInfo uuri = (RawServiceInfo) u;

                return new EqualsBuilder().append(turi.getId(), uuri.getId()).append(turi.getPayload(), uuri.getPayload()).isEquals();
            }
            if (t instanceof BaseServiceInfo && u instanceof BaseServiceInfo) {
                BaseServiceInfo turi = (BaseServiceInfo) t;
                BaseServiceInfo uuri = (BaseServiceInfo) u;

                return Objects.equals(turi.getId(), uuri.getId());
            }

            throw new UnsupportedOperationException("don't know how to compare " + t.getClass().getSimpleName());
        }
        return false;
    }

    static int hashCode(ServiceInfo si) {
        if (si instanceof UriBasedServiceInfo) {
            UriBasedServiceInfo usi = (UriBasedServiceInfo) si;
            return Objects.hash(usi.getId(), usi.getUri());
        }
        if (si instanceof RawServiceInfo) {
            RawServiceInfo rsi = (RawServiceInfo) si;
            return Objects.hash(rsi.getId(), Arrays.hashCode(rsi.getPayload()));
        }
        if (si instanceof BaseServiceInfo) {
            BaseServiceInfo bsi = (BaseServiceInfo) si;
            return Objects.hash(bsi.getId());
        }
        throw new UnsupportedOperationException("don't know how to calculate hashcode " + si.getClass().getSimpleName());
    }

    static Optional<String> getQueryParam(String key, PlainServiceInfo info) {
        QueryStringDecoder decoder = new QueryStringDecoder(info.getQuery(), false);
        for (Entry<String, List<String>> entry : decoder.parameters().entrySet()) {
            if (entry.getKey().equals(key)) {
                List<String> list = entry.getValue();
                String toReturn = Iterables.getOnlyElement(list);
                return Optional.of(toReturn);
            }
        }
        return Optional.empty();
    }

    static String getRequiredQueryParam(String key, PlainServiceInfo info) throws Exception {
        Optional<String> opt = getQueryParam(key, info);
        Preconditions.checkArgument(opt.isPresent(), "param '%s' is not found or miss-configured", key);
        return opt.get();
    }
}
