package com.turbospaces.boot;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.security.KeyStore;
import java.util.Map;
import java.util.Optional;
import java.util.SortedMap;

import javax.net.ssl.SSLContext;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.cloud.DynamicCloud;
import org.springframework.cloud.SmartCloud;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.context.ConfigurableApplicationContext;

import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheck.Result;
import com.codahale.metrics.health.HealthCheckFilter;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.executor.DefaultPlatformExecutorService;
import com.turbospaces.executor.PlatformExecutorService;
import com.turbospaces.ssl.SSL;
import com.turbospaces.ssl.SelfSignedCertificateGenerator;
import com.turbospaces.ups.UPSs;

import io.github.resilience4j.circuitbreaker.CircuitBreakerRegistry;
import io.github.resilience4j.ratelimiter.RateLimiterRegistry;
import io.github.resilience4j.retry.RetryRegistry;
import io.github.resilience4j.timelimiter.TimeLimiterRegistry;
import io.micrometer.core.instrument.composite.CompositeMeterRegistry;
import io.opentracing.Tracer;
import reactor.core.publisher.Flux;

public interface Bootstrap extends SmartCloud, UPSs {
    //
    // life-cycle
    //
    ConfigurableApplicationContext run(String... args);
    void shutdown() throws Exception; // ~ graceful shutdown
    void exit(int code); // ~ exit VM manually

    DynamicCloud cloud();
    String spaceName();
    String appId();
    String release();

    boolean isDevMode();
    default boolean isProdMode() {
        return BooleanUtils.isFalse(isDevMode());
    }

    PlatformScheduledExecutorService globalPlatform();
    PlatformExecutorService platform(String name);
    Map<String, DefaultPlatformExecutorService> platforms();

    ApplicationProperties props();
    ApplicationConfig cfg();

    KeyStore keyStore();
    Tracer tracer();

    void refreshCfg() throws Exception;
    void squashLogging() throws Exception;

    //
    // ~ port
    //
    int port();
    int secondaryPort();
    int tertiaryPort();

    //
    // health-checks and metrics support
    //
    CompositeMeterRegistry meterRegistry();
    void registerHealthCheck(String name, HealthCheck check);
    HealthCheckRegistry healthCheckRegistry();

    RetryRegistry retryRegistry();
    RateLimiterRegistry rateLimiterRegistry();
    TimeLimiterRegistry timeLimiterRegistry();
	CircuitBreakerRegistry circuitBreakerRegistry();

    @Override
    default void addUps(ServiceInfo info) {
        cloud().addUps(info);
    }
    @Override
    default boolean removeUps(String id) {
        return cloud().removeUps(id);
    }
    @Override
    default boolean removeUps(ServiceInfo si) {
        return cloud().removeUps(si);
    }
    @Override
    default <T extends ServiceInfo> Flux<ServiceInfo> serviceInfoByName(String ups) {
        return cloud().serviceInfoByName(ups);
    }
    @Override
    default <T extends ServiceInfo> Flux<ServiceInfo> scopedServiceInfoByName(String scope, String name) {
        return cloud().scopedServiceInfoByName(scope, name);
    }
    default SortedMap<String, Result> healthCheck() {
        return healthCheckRegistry().runHealthChecks(new HealthCheckFilter() {
            @Override
            public boolean matches(String name, HealthCheck healthCheck) {
                if (healthCheck instanceof AbstractHealtchCheck) {
                    AbstractHealtchCheck check = (AbstractHealtchCheck) healthCheck;
                    return check.isPermanent();
                }
                return true;
            }
        });
    }

    default void createSelfSignedKeyStoreIfAbsent(File file) throws Exception {
        createSelfSignedKeyStoreIfAbsent(file, SelfSignedCertificateGenerator.PASSWORD);
    }
    default void createSelfSignedKeyStoreIfAbsent(File file, String password) throws Exception {
        if (file.exists()) {} else {
            SelfSignedCertificateGenerator ssc = new SelfSignedCertificateGenerator();
            ssc.setBootstrap(this);
            KeyStore keystore = ssc.call();
            try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
                keystore.store(out, password.toCharArray());
                FileUtils.writeByteArrayToFile(file, out.toByteArray());
            }
        }
    }
    default Optional<SSLContext> generateSelfSignedCertificate() throws Exception {
        if (props().APP_USE_SELF_SIGNED_CERTIFICATE.get()) {
            // command to export cert from ca, may be helpful when testing smth in safari locally
            // keytool -export -keystore frontend_keystore -alias uam-frontend -file frontend.cer
            if (isDevMode()) {
                File keystoreFile = new File(FileUtils.getUserDirectory().getAbsolutePath() + "/self_signed_keystore");
                createSelfSignedKeyStoreIfAbsent(keystoreFile);

                SSL ssl = new SSL();
                ssl.loadKeyStore(keystoreFile, SelfSignedCertificateGenerator.PASSWORD);
                return Optional.of(ssl.build());
            }
        }
        return Optional.empty();
    }
}
