package com.turbospaces.cfg;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Range;
import com.google.common.net.HttpHeaders;
import com.netflix.archaius.api.Property;
import com.netflix.archaius.config.DefaultCompositeConfig;
import com.netflix.archaius.config.DefaultSettableConfig;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.logging.ClassUtils;

import reactor.core.scheduler.Scheduler;
import reactor.util.retry.Retry;

public class ApplicationProperties implements TypedPropertyFactory {
    private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationProperties.class);

    public static final int PRIMARY_PORT = 8089;
    public static final int SECONDARY_PORT = 8091;
    public static final int TERTIARY_PORT = 8093;

    private final DynamicPropertyFactory pf;

    public ApplicationProperties(ApplicationConfig cfg) {
        pf = cfg.factory();

        //
        // cloud
        //
        CLOUD_APP_ID = pf.get(CloudOptions.CLOUD_APP_ID, String.class).orElse(System.getProperty("user.name"));
        CLOUD_APP_INSTANCE_INDEX = pf.get(CloudOptions.CLOUD_APP_INSTANCE_INDEX, String.class).orElse("0");
        CLOUD_APP_NAME = pf.get(CloudOptions.CLOUD_APP_NAME, String.class);
        CLOUD_APP_SPACE_NAME = pf.get(CloudOptions.CLOUD_APP_SPACE_NAME, String.class);
        CLOUD_APP_HOST = pf.get(CloudOptions.CLOUD_APP_HOST, String.class).orElse("localhost");
        CLOUD_APP_PORT = pf.get(CloudOptions.CLOUD_APP_PORT, int.class).orElse(PRIMARY_PORT);
        CLOUD_APP_SECONDARY_PORT = pf.get(CloudOptions.CLOUD_APP_SECONDARY_PORT, int.class).orElse(SECONDARY_PORT);
        CLOUD_APP_TERTIARY_PORT = pf.get(CloudOptions.CLOUD_APP_TERTIARY_PORT, int.class).orElse(TERTIARY_PORT);

        //
        // ~ CACHE
        //
        CACHE_DEFAULT_MAX_TTL = pf.get("cache.default.max-ttl", Duration.class).orElse(Duration.ofMinutes(30));
        CACHE_DEFAULT_MAX_IDLE = pf.get("cache.default.max-idle", Duration.class).orElse(Duration.ofMinutes(1));
        CACHE_DEFAULT_MAX_SIZE = pf.get("cache.default.max-size", int.class).orElse(1024 * 16);
        CACHE_LOCAL_NEVER_EXPIRE = pf.get("cache.local.never-expire", boolean.class).orElse(false);
        CACHE_REPLICATED_NEVER_EXPIRE = pf.get("cache.replicated.never-expire", boolean.class).orElse(true);

        //
        // ~ TCP
        //
        TCP_REUSE_ADDRESS = pf.get("tcp.reuse-address", boolean.class).orElse(true);
        TCP_NO_DELAY = pf.get("tcp.no-delay", boolean.class).orElse(true);
        TCP_KEEP_ALIVE = pf.get("tcp.keep-alive", boolean.class).orElse(true);
        TCP_KEEP_ALIVE_TIMEOUT = rangeValue("tcp.keep-alive.timeout", Duration.ofMinutes(1), Range.closed(Duration.ofSeconds(0), Duration.ofMinutes(15)));
        TCP_CONNECTION_TIMEOUT = rangeValue("tcp.connection.timeout", Duration.ofSeconds(15), Range.closed(Duration.ofSeconds(0), Duration.ofMinutes(1)));
        TCP_CONNECTION_VALIDATE_AFTER_INACTIVITY_TIMEOUT = rangeValue(
                "tcp.connection.validate-after-inactivity.timeout",
                Duration.ofSeconds(0),
                Range.closed(Duration.ofSeconds(0), Duration.ofMinutes(1)));
        TCP_CONNECTION_EVICT_IDLE_TIMEOUT = rangeValue(
                "tcp.connection.evict-idle.timeout",
                Duration.ofSeconds(0),
                Range.closed(Duration.ofSeconds(0), Duration.ofMinutes(1)));
        TCP_CONNECTION_CLOSE_IDLE_IMMEDIATELY = pf.get("tcp.connection.close-idle-immediately", boolean.class).orElse(false);
        TCP_SOCKET_TIMEOUT = rangeValue("tcp.socket.timeout", Duration.ofSeconds(60), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        TCP_SOCKET_BACKLOG = pf.get("tcp.socket.backlog", int.class).orElse(1024);
        TCP_FRAME_MAX_SIZE = pf.get("tcp.frame.max-size", int.class).orElse(1024 * 1024 * 16); // 16MB

        //
        // ~ NETTY
        //
        NETTY_ACCEPTOR_POOL_SIZE = pf.get("netty.acceptor-pool.size", int.class).orElse(Runtime.getRuntime().availableProcessors());
        NETTY_WORKER_POOL_SIZE = pf.get("netty.worker-pool.size", int.class).orElse(32);
        NETTY_VERSION_SEND_HTTP_HEADER = pf.get("netty.version.send-http-header", boolean.class).orElse(false);
        NETTY_VERSION_HTTP_HEADER_NAME = pf.get("netty.version.http-header-name", String.class).orElse("X-Netty-Version");

        //
        // ~ JETTY
        //
        JETTY_POOL_MIN_SIZE = pf.get("jetty.pool.min-size", int.class).orElse(32);
        JETTY_POOL_MAX_SIZE = pf.get("jetty.pool.max-size", int.class).orElse(64);
        JETTY_POOL_QUEUE_MAX_SIZE = pf.get("jetty.pool.queue-max-size", int.class).orElse(16 * 1024);
        JETTY_POOL_MAX_IDLE = pf.get("jetty.pool.max-idle", Duration.class).orElse(Duration.ofMinutes(1));
        JETTY_HTTP_SEND_DATE_HEADER = pf.get("jetty.http.send-date-header", boolean.class).orElse(true);
        JETTY_GZIP_ENABLED = pf.get("jetty.gzip.enabled", boolean.class).orElse(true);

        //
        // ~ CORS
        //
        // @formatter:off
        CORS_ALLOWED_ORIGINS = listOfStrings("cors.allowed-origins");
        CORS_ALLOWED_HEADERS = listOfStrings("cors.allowed-headers").orElse(Lists.newArrayList("access-control-allow-origin", "content-type", "sentry-trace", "x-platform", "x-trace-id"));
        CORS_ALLOWED_METHODS = listOfStrings("cors.allowed-methods").orElse(Lists.newArrayList("GET", "POST", "DELETE", "PUT", "PATCH"));
        CORS_EXPOSED_HEADERS = listOfStrings("cors.exposed-headers").orElse(Lists.newArrayList("location"));
        CORS_ALLOW_CREDENTIALS = pf.get("cors.allow-credentials", boolean.class).orElse(true);
        CORS_MAX_AGE = rangeValue("cors.max-age", Duration.ofMinutes(1), Range.closed(Duration.ofSeconds(15), Duration.ofHours(1)));
        // @formatter:on

        //
        // ~ JDBC
        //
        JDBC_DEFAULT_INSTANCE_REGISTER = pf.get("jdbc.default-instance.register", boolean.class).orElse(false);
        JDBC_PERSIST_BATCH_SIZE = pf.get("jdbc.persist-batch.size", int.class).orElse(100);
        JDBC_QUERY_BATCH_SIZE = pf.get("jdbc.query-batch.size", int.class).orElse(250);
        JDBC_LAZY_BATCH_SIZE = pf.get("jdbc.lazy-batch.size", int.class).orElse(100);
        JDBC_LAZY_SEQUENCE_BATCH_SIZE = pf.get("jdbc.lazy-sequence-batch.size", int.class).orElse(100);
        JDBC_POOL_MIN_SIZE = pf.get("jdbc.pool.min-size", int.class).orElse(1);
        JDBC_POOL_MAX_SIZE = pf.get("jdbc.pool.max-size", int.class).orElse(10);
        JDBC_READ_ONLY_POOL_MIN_SIZE = pf.get("jdbc.read-only-pool.min-size", int.class).orElse(1);
        JDBC_READ_ONLY_POOL_MAX_SIZE = pf.get("jdbc.read-only-pool.max-size", int.class).orElse(5);
        JDBC_CONNECTION_TIMEOUT = rangeValue("jdbc.connection.timeout", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(1)));
        JDBC_LEAK_DETECTION_TIMEOUT = rangeValue("jdbc.leak-detection.timeout", Duration.ofSeconds(5), Range.closed(Duration.ZERO, Duration.ofMinutes(30)));

        //
        // ~ KAFKA
        //
        KAFKA_MAX_BLOCK = rangeValue("kafka.max-block", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(1)));
        KAFKA_IDEMPOTENCE_ENABLE = pf.get("kafka.idempotence.enable", boolean.class).orElse(true);
        KAFKA_ACKS = pf.get("kafka.acks", String.class).orElse("all");
        KAFKA_COMPRESSION_TYPE = pf.get("kafka.compression.type", String.class).orElse("none");
        KAFKA_RECORD_MAX_REQUEST_SIZE = pf.get("kafka.record.max-request-size", int.class).orElse(4 * 1024 * 1024); // 4 MB
        KAFKA_AUTO_OFFSET_RESET = pf.get("kafka.auto.offset.reset", String.class);
        KAFKA_HEARTBEAT_INTERVAL = rangeValue("kafka.heartbeat.interval", Duration.ofSeconds(5), Range.closed(Duration.ofSeconds(1), Duration.ofSeconds(15)));
        KAFKA_SESSION_TIMEOUT = rangeValue("kafka.session.timeout", Duration.ofSeconds(30), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(1)));
        KAFKA_POLL_MAX_INTERVAL = rangeValue("kafka.poll.max-interval", Duration.ofMinutes(10), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(30)));
        KAFKA_POLL_MAX_RECORDS = pf.get("kafka.poll.max-records", int.class).orElse(1024);
        KAFKA_MIN_WORKERS = pf.get("kafka.min-workers", int.class).orElse(0);
        KAFKA_MAX_WORKERS = pf.get("kafka.max-workers", int.class).orElse(128);
        KAFKA_MAX_POLL_CONCURRENCY = pf.get("kafka.max-poll-concurrency", int.class).orElse(2);
        KAFKA_NACK_ON_QUEUE_FULL = pf.get("kafka.nack.on-queue-full", boolean.class).orElse(false);
        KAFKA_SYSTEM_EXIT_ON_QUEUE_FULL = pf.get("kafka.system-exit.on-queue-full", boolean.class).orElse(true);
        KAFKA_SYSTEM_EXIT_CODE = pf.get("kafka.system-exit.code", int.class).orElse(-1);
        KAFKA_SYSTEM_EXIT_DELAY = rangeValue("kafka.system-exit.delay", Duration.ofSeconds(5), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        BATCH_COMPLETION_TIMEOUT = rangeValue("batch.completion.timeout", Duration.ofMinutes(2), Range.closed(Duration.ofSeconds(30), Duration.ofMinutes(5)));
        KAFKA_SPRING_BACK_OFF_ENABLED = pf.get("kafka.spring-back-off.enabled", boolean.class).orElse(true);

        //
        // ~ GRPC
        //
        GRPC_MIN_WORKERS = pf.get("grpc.min-workers", int.class).orElse(1);
        GRPC_MAX_WORKERS = pf.get("grpc.max-workers", int.class).orElse(1024);

        //
        // ~ QUARTZ
        //
        QUARTZ_SCHEDULER_ID = pf.get("quartz.scheduler.id", String.class).orElse("AUTO");
        QUARTZ_WORKER_POOL_COUNT = pf.get("quartz.worker-pool.count", Integer.class).orElse(10);
        QUARTZ_JOBSTORE_TABLE_PREFIX = pf.get("quartz.jobstore.table-prefix", String.class).orElse("qrtz_");
        QUARTZ_JOBSTORE_USE_PROPS = pf.get("quartz.jobstore.use-props", Boolean.class).orElse(false);
        QUARTZ_JOBSTORE_ACQUIRE_TRIGGERS_WITHIN_LOCK = pf.get("quartz.jobstore.acquire-triggers-within-lock", Boolean.class).orElse(false);
        QUARTZ_JOBSTORE_IS_CLUSTERED = pf.get("quartz.jobstore.is-clustered", Boolean.class).orElse(false);
        QUARTZ_CONNECTION_POOL_MIN = pf.get("quartz.connection-pool.min", Integer.class).orElse(1);
        QUARTZ_CONNECTION_POOL_MAX = pf.get("quartz.connection-pool.max", Integer.class).orElse(5);
        QUARTZ_INSTANCE_ID = pf.get("quartz.instance.id", String.class);
        QUARTZ_SHUTDOWN_WAIT_FOR_JOBS_COMPLETION = pf.get("quartz.shutdown.wait-for-jobs-completion", boolean.class);
        QUARTZ_ENFORCE_DISABLED_JOBS_ENABLED = pf.get("quartz.enforce-disabled-jobs.enabled", boolean.class).orElse(true);

        //
        // ~ APP
        //
        APP_DEV_MODE = pf.get("app.dev.mode", boolean.class).orElse(true);
        APP_DNS_QUERY = pf.get("app.dns.query", String.class);
        APP_BACKOFF_RETRY_FIRST = rangeValue("app.backoff-retry.first", Duration.ofSeconds(1), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_BACKOFF_RETRY_MAX = rangeValue("app.backoff-retry.max", Duration.ofMinutes(1), Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(15)));
        APP_BACKOFF_RETRY_NUM = pf.get("app.backoff-retry.num", int.class).orElse(10);
        APP_SENTRY_ENABLED = pf.get("app.sentry.enabled", boolean.class).orElse(true);
        APP_USE_SELF_SIGNED_CERTIFICATE = pf.get("app.use.self-signed-certificate", boolean.class).orElse(false);
        APP_LOGGING_RESET_TO = pf.get("app.logging.reset-to", String.class);
        APP_LOGGING_DRY_RUN = pf.get("app.logging.dry-run", boolean.class).orElse(false);
        APP_SPRING_INTEGRATIONS_LOGGING_ENABLED = pf.get("app.spring-integrations.logging.enabled", boolean.class).orElse(false);
        APP_METRICS_DRY_RUN = pf.get("app.metrics.dry-run", boolean.class).orElse(false);
        APP_METRICS_ELK_REPORTER_ENABLED = pf.get("app.metrics.elk-reporter.enabled", boolean.class).orElse(false);
        APP_METRICS_INFLUX_REPORTER_ENABLED = pf.get("app.metrics.influx-reporter.enabled", boolean.class).orElse(true);
        APP_METRICS_PROMETHEUS_REPORTER_ENABLED = pf.get("app.metrics.prometheus-reporter.enabled", boolean.class).orElse(true);
        APP_ALERTS_DRY_RUN = pf.get("app.alerts.dry-run", boolean.class).orElse(false);
        APP_EXTERNAL_IP = pf.get("app.external.ip", String.class);
        APP_JMX_DOMAIN = pf.get("app.jmx.domain", String.class).orElse("metrics");
        APP_DB_MIGRATION_ENABLED = pf.get("app.db-migration.enabled", boolean.class).orElse(true);
        APP_DB_MIGRATION_BASELINE = pf.get("app.db-migration.baseline", String.class);
        APP_DB_MIGRATION_PATH = pf.get("app.db-migration.path", String.class).orElse("db/sql-migration");
        APP_PLATFORM_POOL_SIZE = pf.get("app.platform.pool-size", int.class).orElse(64);
        APP_DYNAMIC_PLATFORM_DEFAULT_MAX_SIZE = pf.get("app.dynamic-platform.default-max-size", int.class).orElse(1024);
        APP_PLATFORM_MAX_IDLE = rangeValue("app.platform.max-idle", Duration.ofSeconds(1), Range.closed(Duration.ofSeconds(1), Duration.ofHours(1)));
        APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT = rangeValue(
                "app.platform.graceful-shutdown-timeout",
                Duration.ofSeconds(30),
                Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_TIMER_INTERVAL = rangeValue("app.timer.interval", Duration.ofSeconds(15), Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_METRICS_BULK_SIZE = pf.get("app.metrics.bulk-size", int.class).orElse(1024);
        APP_HEAP_DUMP_ON_OOM_ENABLED = pf.get("app.heap-dump-on-oom.enabled", boolean.class).orElse(true);
        APP_HEAP_DUMP_DIR = pf.get("app.heap-dump.dir", String.class).orElse(System.getProperty("java.io.tmpdir"));
        APP_WAIT_FOR_HEALTHCHECKS_INTERVAL = rangeValue(
                "app.wait-for-healthchecks.interval",
                Duration.ofSeconds(1),
                Range.closed(Duration.ofSeconds(1), Duration.ofMinutes(1)));
        APP_WAIT_FOR_HEALTHCHECKS_ENABLED = pf.get("app.wait-for-healthchecks.enabled", boolean.class).orElse(false);
        APP_WAIT_FOR_HEALTHCHECKS_TIMEOUT = rangeValue(
                "app.wait-for-healthchecks.timeout",
                Duration.ofSeconds(60),
                Range.closed(Duration.ofSeconds(15), Duration.ofMinutes(5)));
        APP_DATA_START_CLEAN = pf.get("app.data.start-clean", boolean.class).orElse(false);
        APP_SHUTDOWN_HOOK_ENABLED = pf.get("app.shutdown-hook.enabled", boolean.class).orElse(true);
        APP_CLEAR_CONFIG_AT_SHUTDOWN_ENABLED = pf.get("app.clear-config-at-shutdown.enabled", boolean.class).orElse(false);
        APP_CFG_DYNAMIC_POLLING_ENABLED = pf.get("app.cfg-dynamic-polling.enabled", boolean.class).orElse(true);
        DYNAMIC_PROPERTY_KEYS = listOfStrings("dynamic-property-keys").orElse(List.of());
        API_RATE_LIMIT_OUTGOING_DEFAULT_PERIOD = pf.get("api.rate.limit.outgoing-default.period", Duration.class).orElse(Duration.ofSeconds(30));
        API_RATE_LIMIT_OUTGOING_DEFAULT_COUNT = pf.get("api.rate.limit.outgoing-default.count", int.class).orElse(3000);
        API_RATE_LIMIT_OUTGOING_DEFAULT_TIMEOUT = pf.get("api.rate.limit.outgoing-default.timeout", Duration.class).orElse(Duration.ofSeconds(2));

        API_RATE_LIMIT_ENABLED = pf.get("api.rate.limit.enabled", boolean.class).orElse(true);
        API_RATE_LIMIT_PERIOD = pf.get("api.rate.limit.period", Duration.class).orElse(Duration.ofMinutes(1));
        API_RATE_LIMIT_COUNT = pf.get("api.rate.limit.count", int.class).orElse(6000);
        API_RATE_LIMIT_TIMEOUT = pf.get("api.rate.limit.timeout", Duration.class).orElse(Duration.ofSeconds(0));

        CONNECTION_USER_AGENT_ACCEPT_EMPTY = pf.get("connection.user-agent.accept-empty", boolean.class).orElse(true);
        CONNECTION_CHALLENGE_ENABLED = pf.get("connection.challenge.enabled", boolean.class).orElse(true);
        CONNECTION_CHALLENGE_ACCEPT_EMPTY = pf.get("connection.challenge.accept-empty", boolean.class).orElse(true);
        CONNECTION_CHALLENGE_MAX_TIMESTAMP_DIFF = rangeValue(
                "connection.challenge.max-timestamp-diff",
                Duration.ofMinutes(5),
                Range.closed(Duration.ofSeconds(30), Duration.ofMinutes(30)));

        //
        // ~ HTTP
        //
        HTTP_POOL_MAX_PER_ROUTE = pf.get("http.pool.max-per-route", int.class).orElse(32);
        HTTP_POOL_MAX_SIZE = pf.get("http.pool.max-size", int.class).orElse(256);
        HTTP_METRICS_LATENCY_KEY = pf.get("http.metrics.latency-key", String.class).orElse("server-latency");
        HTTP_RATE_BARRIER_ACCEPTED_KEY = pf.get("http.rate-barrier.accepted-key", String.class).orElse("http.rate-barrier.accepted");
        HTTP_RATE_BARRIER_DEBOUNCED_KEY = pf.get("http.rate-barrier.debounced-key", String.class).orElse("http.rate-barrier.debounced");
        HTTP_RETRY_REQUEST_ENABLED = pf.get("http.retry-request.enabled", boolean.class).orElse(false);
        HTTP_RETRY_REQUEST_COUNT = pf.get("http.retry-request.count", int.class).orElse(3);
        HTTP_RETRY_ALREADY_SENT_REQUEST_ENABLED = pf.get("http.retry-already-sent-request.enabled", boolean.class).orElse(false);
        HTTP_HEADERS_TO_MASK = listOfStrings("http.headers.to-mask").orElse(Lists.newArrayList(HttpHeaders.AUTHORIZATION, "X-Api-Key"));
        HTTP_COOKIES_TO_MASK = listOfStrings("http.cookies.to-mask");
        HTTP_QUERY_PARAMS_TO_MASK = listOfStrings("http.query-params.to-mask").orElse(Lists.newArrayList("password"));
        HTTP_METRICS_INBOUND_PATH_MASK = listOfPatterns("http.metrics.inbound-path-mask");
        HTTP_METRICS_OUTBOUND_PATH_MASK = listOfPatterns("http.metrics.outbound-path-mask");
        HTTP_PROXY = pf.get("http.proxy", String.class);
        HTTP_REQUEST_TO_PROXY_PATTERNS = listOfPatterns("http.request-to-proxy.patterns");
        MOCK_HTTP_PROXY = pf.get("mock.http.proxy", String.class);
        MOCK_HTTP_REQUEST_TO_PROXY_PATTERNS = listOfPatterns("mock.http.request-to-proxy.patterns");
        SKIP_PAYLOAD_CHECKSUM_CHECK = pf.get("skip-payload-checksum-check", boolean.class).orElse(false);

        //
        // ~ PubSub
        //
        PUBSUB_SSL_ENABLED = pf.get("pubsub.ssl.enabled", boolean.class).orElse(true);
        PUBSUB_ACK_DEADLINE = pf.get("pubsub.ack-deadline", Duration.class).orElse(Duration.ofSeconds(60));
        PUBSUB_REQUEST_REPLY_TIMEOUT = pf.get("pubsub.request-reply.timeout", Duration.class).orElse(Duration.ofSeconds(30));
        PUBSUB_POLLING_ENABLED = pf.get("pubsub.polling.enabled", boolean.class).orElse(false);

        //
        // ~ Sentry
        //
        SENTRY_ALERTS_FILTER_ENABLED = pf.get("sentry.alerts-filter.enabled", boolean.class).orElse(true);
        SENTRY_ALERTS_EX_CLASSES_TO_IGNORE = pf.getProperty("sentry.alerts.ex.classes.to.ignore").asType(s -> ClassUtils.parseThrowableClasses(s, ","), "");
        SENTRY_ALERTS_EX_MSGS_TO_IGNORE = pf.getProperty("sentry.alerts.ex.msgs.to.ignore").asType(s -> splitString(s, "~"), "");
        SENTRY_ALERTS_EX_PATTERNS_TO_IGNORE = pf.getProperty("sentry.alerts.ex.patterns.to.ignore").asType(s -> splitString(s, "~"), "");
        SENTRY_ALERTS_LOG_MESSAGES_TO_IGNORE = pf.getProperty("sentry.alerts.log.messages.to.ignore").asType(s -> splitString(s, "~"), "");
        SENTRY_ALERTS_LOGGERS_TO_IGNORE = pf.getProperty("sentry.alerts.loggers.to.ignore").asType(s -> splitString(s, "~"), "");
        REPORT_TO_SENTRY_ON_TIMEOUT_ERROR = pf.get("report-to-sentry.on-timeout-error", boolean.class).orElse(true);
    }

    @Override
    public DynamicPropertyFactory factory() {
        return pf;
    }

    @Override
    public Retry retry(Scheduler scheduler) {
        int numRetries = APP_BACKOFF_RETRY_NUM.get();
        Duration firstBackoff = APP_BACKOFF_RETRY_FIRST.get();
        Duration maxBackoff = APP_BACKOFF_RETRY_MAX.get();
        return Retry.backoff(numRetries, firstBackoff).maxBackoff(maxBackoff).scheduler(scheduler);
    }

    @Override
    public boolean warnInconsistency(String... cfgs) throws Exception {
        boolean toReturn = false;

        ImmutableList<String> l = ArrayUtils.isEmpty(cfgs) ? ImmutableList.of(APPLICATION_PROPERTIES) : ImmutableList.copyOf(cfgs);
        DefaultCompositeConfig gitComposeConfig = (DefaultCompositeConfig) pf.getConfig().getConfig(DynamicCompositeConfig.GIT_CFG_NAME);
        Map<String, Object> map = readFieldAsMap();

        for (String next : l) {
            Path path = Paths.get(CLOUD_APP_ID.get(), next);
            DefaultSettableConfig config = (DefaultSettableConfig) gitComposeConfig.getConfig(path.toString());

            if (Objects.nonNull(config)) {
                Iterator<String> it = config.getKeys();
                List<String> unknowns = Lists.newArrayList();

                while (it.hasNext()) {
                    String key = it.next();

                    if (!map.containsKey(key) && !DYNAMIC_PROPERTY_KEYS.get().contains(key)) {
                        unknowns.add(key);
                        toReturn = true;
                    }
                }

                if (BooleanUtils.isFalse(unknowns.isEmpty())) {
                    LOGGER.error("unknown keys: {} in {}", unknowns, map.keySet());
                }
            }
        }

        return toReturn;
    }

    @Override
    public String externalIp() {
        Property<String> prop = APP_EXTERNAL_IP.orElse(ipSupplier.get());
        return prop.get();
    }

    protected List<Pattern> parsePatterns(String patterns) {
        try {
            return patterns.isEmpty()
                    ? Collections.emptyList()
                    : Arrays.stream(patterns.split(",")).map(Pattern::compile).collect(Collectors.toList());
        } catch (Exception e) {
            LOGGER.error("Error on patter init {}", patterns, e);
            return Collections.emptyList();
        }
    }

    protected List<String> splitString(String value, String separator) {
        try {
            return value.trim().isEmpty() ? Collections.emptyList() : Arrays.asList(value.split(separator));
        } catch (Exception e) {
            LOGGER.error("Error on string list parse {}", value);
            return Collections.emptyList();
        }
    }

    //
    // ~ cloud
    //
    public final Property<String> CLOUD_APP_ID;
    public final Property<String> CLOUD_APP_SPACE_NAME;
    public final Property<String> CLOUD_APP_NAME;
    public final Property<String> CLOUD_APP_INSTANCE_INDEX;
    public final Property<String> CLOUD_APP_HOST;
    public final Property<Integer> CLOUD_APP_PORT;
    public final Property<Integer> CLOUD_APP_SECONDARY_PORT;
    public final Property<Integer> CLOUD_APP_TERTIARY_PORT;

    //
    // ~ CACHE
    //
    public final Property<Duration> CACHE_DEFAULT_MAX_TTL;
    public final Property<Duration> CACHE_DEFAULT_MAX_IDLE;
    public final Property<Integer> CACHE_DEFAULT_MAX_SIZE;
    public final Property<Boolean> CACHE_REPLICATED_NEVER_EXPIRE;
    public final Property<Boolean> CACHE_LOCAL_NEVER_EXPIRE;

    //
    // ~ TCP
    //
    public final Property<Boolean> TCP_REUSE_ADDRESS;
    public final Property<Boolean> TCP_NO_DELAY;
    public final Property<Boolean> TCP_KEEP_ALIVE;
    public final Property<Duration> TCP_KEEP_ALIVE_TIMEOUT;
    public final Property<Duration> TCP_CONNECTION_TIMEOUT;
    public final Property<Duration> TCP_CONNECTION_VALIDATE_AFTER_INACTIVITY_TIMEOUT;
    public final Property<Duration> TCP_CONNECTION_EVICT_IDLE_TIMEOUT;
    public final Property<Boolean> TCP_CONNECTION_CLOSE_IDLE_IMMEDIATELY;

    public final Property<Duration> TCP_SOCKET_TIMEOUT;
    public final Property<Integer> TCP_SOCKET_BACKLOG;
    public final Property<Integer> TCP_FRAME_MAX_SIZE;
    public final Property<Integer> HTTP_POOL_MAX_SIZE;

    //
    // ~ NETTY
    //
    public final Property<Integer> NETTY_ACCEPTOR_POOL_SIZE;
    public final Property<Integer> NETTY_WORKER_POOL_SIZE;
    public final Property<Boolean> NETTY_VERSION_SEND_HTTP_HEADER;
    public final Property<String> NETTY_VERSION_HTTP_HEADER_NAME;

    //
    // ~ JETTY
    //
    public final Property<Integer> JETTY_POOL_MAX_SIZE;
    public final Property<Integer> JETTY_POOL_MIN_SIZE;
    public final Property<Integer> JETTY_POOL_QUEUE_MAX_SIZE;
    public final Property<Duration> JETTY_POOL_MAX_IDLE;
    public final Property<Boolean> JETTY_HTTP_SEND_DATE_HEADER;
    public final Property<Boolean> JETTY_GZIP_ENABLED;

    //
    // ~ CORS
    //
    public final Property<List<String>> CORS_ALLOWED_ORIGINS;
    public final Property<List<String>> CORS_ALLOWED_HEADERS;
    public final Property<List<String>> CORS_ALLOWED_METHODS;
    public final Property<List<String>> CORS_EXPOSED_HEADERS;
    public final Property<Boolean> CORS_ALLOW_CREDENTIALS;
    public final Property<Duration> CORS_MAX_AGE;

    //
    // ~ JDBC
    //
    public final Property<Boolean> JDBC_DEFAULT_INSTANCE_REGISTER;
    public final Property<Integer> JDBC_PERSIST_BATCH_SIZE;
    public final Property<Integer> JDBC_QUERY_BATCH_SIZE;
    public final Property<Integer> JDBC_LAZY_BATCH_SIZE;
    public final Property<Integer> JDBC_LAZY_SEQUENCE_BATCH_SIZE;
    public final Property<Integer> JDBC_POOL_MIN_SIZE;
    public final Property<Integer> JDBC_POOL_MAX_SIZE;
    public final Property<Integer> JDBC_READ_ONLY_POOL_MIN_SIZE;
    public final Property<Integer> JDBC_READ_ONLY_POOL_MAX_SIZE;
    public final Property<Duration> JDBC_CONNECTION_TIMEOUT;
    public final Property<Duration> JDBC_LEAK_DETECTION_TIMEOUT;

    //
    // ~ KAFKA
    //
    public final Property<Duration> KAFKA_MAX_BLOCK;
    public final Property<Boolean> KAFKA_IDEMPOTENCE_ENABLE;
    public final Property<String> KAFKA_ACKS;
    public final Property<String> KAFKA_COMPRESSION_TYPE;
    public final Property<Integer> KAFKA_RECORD_MAX_REQUEST_SIZE;
    public final Property<String> KAFKA_AUTO_OFFSET_RESET;
    public final Property<Duration> KAFKA_HEARTBEAT_INTERVAL;
    public final Property<Duration> KAFKA_SESSION_TIMEOUT;
    public final Property<Duration> KAFKA_POLL_MAX_INTERVAL;
    public final Property<Integer> KAFKA_POLL_MAX_RECORDS;
    public final Property<Integer> KAFKA_MIN_WORKERS;
    public final Property<Integer> KAFKA_MAX_WORKERS;
    public final Property<Integer> KAFKA_MAX_POLL_CONCURRENCY;
    public final Property<Boolean> KAFKA_SYSTEM_EXIT_ON_QUEUE_FULL;
    public final Property<Boolean> KAFKA_NACK_ON_QUEUE_FULL;
    public final Property<Integer> KAFKA_SYSTEM_EXIT_CODE;
    public final Property<Duration> KAFKA_SYSTEM_EXIT_DELAY;
    public final Property<Duration> BATCH_COMPLETION_TIMEOUT;
    public final Property<Boolean> KAFKA_SPRING_BACK_OFF_ENABLED;

    //
    // ~ GRPC
    //
    public final Property<Integer> GRPC_MIN_WORKERS;
    public final Property<Integer> GRPC_MAX_WORKERS;

    //
    // ~ QUARTZ
    //
    public final Property<String> QUARTZ_SCHEDULER_ID;
    public final Property<Integer> QUARTZ_WORKER_POOL_COUNT;
    public final Property<String> QUARTZ_JOBSTORE_TABLE_PREFIX;
    public final Property<Boolean> QUARTZ_JOBSTORE_USE_PROPS;
    public final Property<Boolean> QUARTZ_JOBSTORE_ACQUIRE_TRIGGERS_WITHIN_LOCK;
    public final Property<Integer> QUARTZ_CONNECTION_POOL_MIN;
    public final Property<Integer> QUARTZ_CONNECTION_POOL_MAX;
    public final Property<Boolean> QUARTZ_JOBSTORE_IS_CLUSTERED;
    public final Property<String> QUARTZ_INSTANCE_ID;
    public final Property<Boolean> QUARTZ_SHUTDOWN_WAIT_FOR_JOBS_COMPLETION;
    public final Property<Boolean> QUARTZ_ENFORCE_DISABLED_JOBS_ENABLED;

    //
    // ~ HTTP
    //
    public final Property<List<Pattern>> HTTP_METRICS_INBOUND_PATH_MASK;
    public final Property<List<Pattern>> HTTP_METRICS_OUTBOUND_PATH_MASK;
    public final Property<String> HTTP_METRICS_LATENCY_KEY;
    public final Property<String> HTTP_RATE_BARRIER_ACCEPTED_KEY;
    public final Property<String> HTTP_RATE_BARRIER_DEBOUNCED_KEY;
    public final Property<Boolean> HTTP_RETRY_REQUEST_ENABLED;
    public final Property<Integer> HTTP_RETRY_REQUEST_COUNT;
    public final Property<Boolean> HTTP_RETRY_ALREADY_SENT_REQUEST_ENABLED;
    public final Property<String> HTTP_PROXY;
    public final Property<Integer> HTTP_POOL_MAX_PER_ROUTE;
    public final Property<List<Pattern>> HTTP_REQUEST_TO_PROXY_PATTERNS;
    public final Property<String> MOCK_HTTP_PROXY;
    public final Property<List<Pattern>> MOCK_HTTP_REQUEST_TO_PROXY_PATTERNS;
    public final Property<List<String>> HTTP_HEADERS_TO_MASK;
    public final Property<List<String>> HTTP_QUERY_PARAMS_TO_MASK;
    public final Property<List<String>> HTTP_COOKIES_TO_MASK;
    public final Property<Boolean> SKIP_PAYLOAD_CHECKSUM_CHECK;

    //
    // ~ PubSub
    //
    public final Property<Boolean> PUBSUB_SSL_ENABLED;
    public final Property<Duration> PUBSUB_ACK_DEADLINE;
    public final Property<Duration> PUBSUB_REQUEST_REPLY_TIMEOUT;
    public final Property<Boolean> PUBSUB_POLLING_ENABLED;

    //
    // ~ APP
    //
    public final Property<Boolean> APP_DEV_MODE;
    public final Property<String> APP_DNS_QUERY;
    public final Property<Duration> APP_BACKOFF_RETRY_FIRST;
    public final Property<Duration> APP_BACKOFF_RETRY_MAX;
    public final Property<Integer> APP_BACKOFF_RETRY_NUM;
    public final Property<Boolean> APP_SENTRY_ENABLED;
    public final Property<Boolean> APP_USE_SELF_SIGNED_CERTIFICATE;
    public final Property<String> APP_LOGGING_RESET_TO;
    public final Property<Boolean> APP_LOGGING_DRY_RUN;
    public final Property<Boolean> APP_SPRING_INTEGRATIONS_LOGGING_ENABLED;
    public final Property<Boolean> APP_METRICS_DRY_RUN;
    public final Property<Boolean> APP_METRICS_ELK_REPORTER_ENABLED;
    public final Property<Boolean> APP_METRICS_INFLUX_REPORTER_ENABLED;
    public final Property<Boolean> APP_METRICS_PROMETHEUS_REPORTER_ENABLED;
    public final Property<Boolean> APP_ALERTS_DRY_RUN;
    public final Property<String> APP_EXTERNAL_IP;
    public final Property<String> APP_JMX_DOMAIN;
    public final Property<Boolean> APP_DB_MIGRATION_ENABLED;
    public final Property<String> APP_DB_MIGRATION_BASELINE;
    public final Property<String> APP_DB_MIGRATION_PATH;
    public final Property<Integer> APP_PLATFORM_POOL_SIZE;
    public final Property<Integer> APP_DYNAMIC_PLATFORM_DEFAULT_MAX_SIZE;
    public final Property<Duration> APP_PLATFORM_MAX_IDLE;
    public final Property<Duration> APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT;
    public final Property<Duration> APP_TIMER_INTERVAL;
    public final Property<Integer> APP_METRICS_BULK_SIZE;
    public final Property<Boolean> APP_HEAP_DUMP_ON_OOM_ENABLED;
    public final Property<String> APP_HEAP_DUMP_DIR;
    public final Property<Boolean> APP_DATA_START_CLEAN;
    public final Property<Duration> APP_WAIT_FOR_HEALTHCHECKS_INTERVAL;
    public final Property<Boolean> APP_WAIT_FOR_HEALTHCHECKS_ENABLED;
    public final Property<Duration> APP_WAIT_FOR_HEALTHCHECKS_TIMEOUT;
    public final Property<Boolean> APP_SHUTDOWN_HOOK_ENABLED;
    public final Property<Boolean> APP_CLEAR_CONFIG_AT_SHUTDOWN_ENABLED;
    public final Property<Boolean> APP_CFG_DYNAMIC_POLLING_ENABLED;
    public final Property<List<String>> DYNAMIC_PROPERTY_KEYS;
    public final Property<Duration> API_RATE_LIMIT_OUTGOING_DEFAULT_PERIOD;
    public final Property<Integer> API_RATE_LIMIT_OUTGOING_DEFAULT_COUNT;
    public final Property<Duration> API_RATE_LIMIT_OUTGOING_DEFAULT_TIMEOUT;

    public final Property<Boolean> API_RATE_LIMIT_ENABLED;
    public final Property<Duration> API_RATE_LIMIT_PERIOD;
    public final Property<Integer> API_RATE_LIMIT_COUNT;
    public final Property<Duration> API_RATE_LIMIT_TIMEOUT;

    public final Property<Boolean> CONNECTION_USER_AGENT_ACCEPT_EMPTY;
    public final Property<Boolean> CONNECTION_CHALLENGE_ENABLED;
    public final Property<Boolean> CONNECTION_CHALLENGE_ACCEPT_EMPTY;
    public final Property<Duration> CONNECTION_CHALLENGE_MAX_TIMESTAMP_DIFF;

    //
    // ~ Sentry
    //
    public final Property<Boolean> SENTRY_ALERTS_FILTER_ENABLED;
    public final Property<List<Class<? extends Throwable>>> SENTRY_ALERTS_EX_CLASSES_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_EX_MSGS_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_EX_PATTERNS_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_LOG_MESSAGES_TO_IGNORE;
    public final Property<List<String>> SENTRY_ALERTS_LOGGERS_TO_IGNORE;
    public final Property<Boolean> REPORT_TO_SENTRY_ON_TIMEOUT_ERROR;

    private final Supplier<String> ipSupplier = Suppliers.memoize(new com.google.common.base.Supplier<String>() {
        @Override
        public String get() {
            return PlatformUtil.fetchExternalIp(ApplicationProperties.this);
        }
    });
}
