package com.turbospaces.rpc;

import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledFuture;
import java.util.function.Supplier;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

import com.google.common.base.Suppliers;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import com.turbospaces.executor.PlatformExecutorService;

import lombok.extern.slf4j.Slf4j;

@SuppressWarnings("serial")
@Slf4j
public class CompletableRequestReplyMapper<K, V> extends ThreadPoolTaskScheduler implements RequestReplyMapper<K, V> {
    protected final Cache<K, SettableFuture<V>> corr;
    protected Supplier<Boolean> reportToSentryOnTimeout = Suppliers.ofInstance(true);
    protected ScheduledFuture<?> cleanUp;

    public CompletableRequestReplyMapper() {
        super();

        corr = CacheBuilder.newBuilder().removalListener(new RemovalListener<K, SettableFuture<V>>() {
            @Override
            public void onRemoval(RemovalNotification<K, SettableFuture<V>> notification) {
                if (Objects.nonNull(notification.getCause())) {
                    String type = notification.getCause().name().toLowerCase().intern();
                    log.trace("onRemoval({}): {} {}", notification.getKey(), type, notification);
                }
            }
        }).expireAfterWrite(Duration.ofHours(1)).build();

        setDaemon(true);
        setRemoveOnCancelPolicy(true);
        setPoolSize(Runtime.getRuntime().availableProcessors());
    }

    @Override
    public void afterPropertiesSet() {
        super.afterPropertiesSet();
        cleanUp = scheduleWithFixedDelay(new Runnable() {
            @Override
            public void run() {
                corr.cleanUp();
            }
        }, Duration.ofMinutes(1));
    }

    @Override
    public void destroy() {
        super.destroy();
        if (Objects.nonNull(cleanUp)) {
            cleanUp.cancel(false);
        }
    }

    @Override
    public SettableFuture<V> acquire(K key, Duration duration) {
        Map<String, String> mdc = MDC.getCopyOfContextMap(); // ~ capture MDC
        StopWatch stopWatch = StopWatch.createStarted();
        SettableFuture<V> toReturn = SettableFuture.create();
        ConcurrentMap<K, SettableFuture<V>> m = corr.asMap();
        if (Objects.nonNull(m.putIfAbsent(key, toReturn))) {
            toReturn.setException(new IllegalArgumentException("duplicate key violation for correlation id: " + key));
        } else {
            ScheduledFuture<?> timerTask = schedule(new Runnable() {
                @Override
                public void run() {
                    SettableFuture<V> tmp = m.remove(key);
                    if (Objects.nonNull(tmp)) {
                        //
                        // ~ only complete when necessary
                        //
                        if (BooleanUtils.isFalse(tmp.isDone())) {
                            PlatformExecutorService.propagete(mdc);
                            try {
                                tmp.setException(new RequestReplyTimeout(duration, key, reportToSentryOnTimeout.get()));
                                log.info("request-reply(m={}) removed subj due to timeout", key);
                            } catch (Exception err) {
                                log.error(err.getMessage(), err);
                            } finally {
                                MDC.clear();
                            }
                        }
                    }
                }
            }, Instant.now().plus(duration));

            toReturn.addListener(new Runnable() {
                @Override
                public void run() {
                    try {
                        log.trace("about to cancel timeout task: {} for key: {}", timerTask, key);
                        timerTask.cancel(false);

                        toReturn.get();
                        stopWatch.stop();
                        m.remove(key);

                        log.debug("request-reply(m={}) completed in {}", key, stopWatch);
                    } catch (Exception err) {

                    }
                }
            }, MoreExecutors.directExecutor());
        }

        return toReturn;
    }
    @Override
    public boolean contains(K corrId) {
        return Objects.nonNull(corr.getIfPresent(corrId));
    }
    @Override
    public void complete(K key, V value) {
        Objects.requireNonNull(key);
        SettableFuture<V> subj = corr.asMap().remove(key);
        if (Objects.nonNull(subj)) {
            setValue(subj, value);
        } else {
            log.trace("no such correlation for key: {}", key);
        }
    }
    @Override
    public void completeExceptionally(K key, Throwable reason) {
        Objects.requireNonNull(key);
        SettableFuture<V> subj = corr.asMap().remove(key);
        if (Objects.nonNull(subj)) {
            setException(subj, reason);
        } else {
            log.trace("no such correlation for key: {}", key);
        }
    }
    @Override
    public void clear() {
        corr.invalidateAll();
    }
    @Override
    public int pendingCount() {
        return corr.asMap().size();
    }
    protected void setValue(SettableFuture<V> subj, V value) {
        subj.set(value);
    }
    protected void setException(SettableFuture<V> subj, Throwable reason) {
        subj.setException(reason);
    }
}
