package com.turbospaces.ups;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.springframework.cloud.service.BaseServiceInfo;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.UriBasedServiceInfo;

import com.turbospaces.boot.Bootstrap;

import reactor.core.publisher.Flux;

public interface UPSs {
    String CFG = "cfg";
    String SENTRY = "sentry";
    String ELASTIC_SEARCH = "elastic-search";
    String INFLUX = "influx";
    String JAEGER = "jaeger";
    String TEMPORAL = "temporal";

    //
    // ~ multiple peers potentially
    //
    String KAFKA = "kafka";

    String QUARTZ_APP = "quartz-app";

    //
    // ~ PG
    //
    String POSTGRES_OWNER = "postgres-owner";
    String POSTGRES_APP = "postgres-app";

    //
    // ~ testing
    //
    String H2_OWNER = "h2-owner";
    String H2_APP = "h2-app";
    String PUBSUB = "pubsub";

    <T extends ServiceInfo> Flux<ServiceInfo> serviceInfoByName(String ups);

    <T extends ServiceInfo> Flux<ServiceInfo> scopedServiceInfoByName(String scope, String name);

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> Optional<T> findServiceInfoByName(Bootstrap boot, String name) {
        List<ServiceInfo> serviceInfos = boot.cloud().getServiceInfos();
        return (Optional<T>) serviceInfos.stream().filter(new Predicate<ServiceInfo>() {
            @Override
            public boolean test(ServiceInfo input) {
                return input.getId().equals(name);
            }
        }).findAny();
    }

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findRequiredServiceInfoByName(Bootstrap bootstrap, String name) {
        return (T) findServiceInfoByName(bootstrap, name).get();
    }

    static <T extends ServiceInfo> Optional<T> findScopedServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<T> opt = findServiceInfoByName(bootstrap, String.format("%s-%s", scope, name));
        if (opt.isPresent()) {
            return opt;
        }
        opt = findServiceInfoByName(bootstrap, name);
        return opt;
    }

    @SuppressWarnings("unchecked")
    static <T extends ServiceInfo> T findScopedRequiredServiceInfoByName(String scope, Bootstrap bootstrap, String name) {
        Optional<ServiceInfo> opt = findServiceInfoByName(bootstrap, String.format("%s-%s", scope, name));
        if (opt.isPresent()) {
            return (T) opt.get();
        }
        opt = findServiceInfoByName(bootstrap, name);
        return (T) opt.get();
    }

    static boolean isEquals(ServiceInfo t, ServiceInfo u) {
        if (t.getClass().equals(u.getClass())) {
            if (t instanceof UriBasedServiceInfo && u instanceof UriBasedServiceInfo) {
                UriBasedServiceInfo turi = (UriBasedServiceInfo) t;
                UriBasedServiceInfo uuri = (UriBasedServiceInfo) u;

                return new EqualsBuilder().append(turi.getId(), uuri.getId()).append(turi.getUri(), uuri.getUri()).isEquals();
            } else if (t instanceof RawServiceInfo && u instanceof RawServiceInfo) {
                RawServiceInfo turi = (RawServiceInfo) t;
                RawServiceInfo uuri = (RawServiceInfo) u;

                return new EqualsBuilder().append(turi.getId(), uuri.getId()).append(turi.getPayload(), uuri.getPayload()).isEquals();
            } else if (t instanceof BaseServiceInfo && u instanceof BaseServiceInfo) {
                BaseServiceInfo turi = (BaseServiceInfo) t;
                BaseServiceInfo uuri = (BaseServiceInfo) u;

                return Objects.equals(turi.getId(), uuri.getId());
            }

            throw new UnsupportedOperationException("don't know how to compare " + t.getClass().getSimpleName());
        }
        return false;
    }

    static int hashCode(ServiceInfo si) {
        if (si instanceof UriBasedServiceInfo) {
            UriBasedServiceInfo usi = (UriBasedServiceInfo) si;
            return Objects.hash(usi.getId(), usi.getUri());
        } else if (si instanceof RawServiceInfo) {
            RawServiceInfo rsi = (RawServiceInfo) si;
            return Objects.hash(rsi.getId(), Arrays.hashCode(rsi.getPayload()));
        } else if (si instanceof BaseServiceInfo) {
            BaseServiceInfo bsi = (BaseServiceInfo) si;
            return Objects.hash(bsi.getId());
        }
        throw new UnsupportedOperationException("don't know how to calculate hashcode " + si.getClass().getSimpleName());
    }
}
