package com.turbospaces.dispatch;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.ListenableFuture;
import com.turbospaces.api.facade.RequestWrapperFacade;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.executor.WorkUnit;
import com.turbospaces.mdc.MdcUtil;
import com.turbospaces.rpc.QueuePostTemplate;

import api.v1.ApiFactory;

public abstract class AbstractServerRequestConsumer extends AbstractBootstrapAware implements ServiceCallAsyncDispatcher {
    protected final QueuePostTemplate<?> postTemplate;
    protected final ApiFactory apiFactory;
    protected Duration timeout;

    protected AbstractServerRequestConsumer(QueuePostTemplate<?> postTemplate, ApiFactory apiFactory) {
        this.apiFactory = apiFactory;
        this.postTemplate = Objects.requireNonNull(postTemplate);
    }
    @Override
    public void setBootstrap(Bootstrap bootstrap) {
        super.setBootstrap(bootstrap);
        this.timeout = bootstrap.props().BATCH_COMPLETION_TIMEOUT.get();
    }
    protected Optional<ListenableFuture<TransactionalRequestOutcome>> logAndAccept(
            RequestWrapperFacade reqw,
            WorkUnit unit,
            CountDownLatch latch) throws Exception {
        String typeUrl = reqw.body().getTypeUrl();

        //
        // ~ set corresponding MDC values
        //
        String operation = PlatformUtil.toLowerUnderscore(typeUrl);
        MdcUtil.setMdc(unit, operation, reqw.headers());

        try {
            logger.info("IN ::: ({}) workUnit: {}", typeUrl, unit);

            //
            // ~ maybe time-outed already
            //
            int origin = reqw.headers().getTimeout();
            boolean outdated = false;
            if (origin > 0) {
                long delta = System.currentTimeMillis() - unit.timestamp();
                if (delta > TimeUnit.SECONDS.toMillis(origin)) {
                    outdated = true;
                }
            }

            if (outdated) {
                latch.countDown();
            } else {
                return Optional.ofNullable(schedule(unit, reqw, latch));
            }
        } catch (Exception err) {
            logger.error(err.getMessage(), err);
            throw err;
        } finally {
            MdcUtil.clearMdc(unit);
        }

        return Optional.empty();
    }
}
