package com.turbospaces.rpc;

import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.FluentFuture;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.google.protobuf.Any;
import com.google.protobuf.Message;

import api.facade.HeadersFacade;
import api.facade.RequestWrapperFacade;
import api.facade.ResponseStatusFacade;
import api.v1.ApplicationException;
import api.v1.ReplyUtil;
import api.v1.ResponseWrapperFacade;
import io.opentracing.Span;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class DefaultApiResponse<RESP extends Message> implements ApiResponse<RESP> {
    private final Class<RESP> type;
    private final FluentFuture<ResponseWrapperFacade> subject;
    private final RequestWrapperFacade requestWrapper;
    private final Span span;

    public DefaultApiResponse(WrappedQueuePost post, Class<RESP> type) {
        this.subject = FluentFuture.from(post);
        this.type = Objects.requireNonNull(type);
        this.requestWrapper = Objects.requireNonNull(post.requestWrapper());
        this.span = Objects.requireNonNull(post.span());
    }
    @VisibleForTesting
    public DefaultApiResponse(ListenableFuture<ResponseWrapperFacade> future, Class<RESP> type, RequestWrapperFacade sreqw, Span span) {
        this.subject = FluentFuture.from(future);
        this.type = Objects.requireNonNull(type);
        this.requestWrapper = Objects.requireNonNull(sreqw);
        this.span = Objects.requireNonNull(span);
    }
    @Override
    public ApiResponseEntity<RESP> get() throws InterruptedException, ExecutionException {
        runGuard(requestWrapper.body().getTypeUrl(), log);
        ResponseWrapperFacade respw = subject.get();
        return new DefaultApiResponseEntity<>(respw, type);
    }
    @Override
    public ApiResponseEntity<RESP> get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        runGuard(requestWrapper.body().getTypeUrl(), log);
        ResponseWrapperFacade respw = subject.get(timeout, unit);
        return new DefaultApiResponseEntity<>(respw, type);
    }
    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
        return subject.cancel(mayInterruptIfRunning);
    }
    @Override
    public boolean isCancelled() {
        return subject.isCancelled();
    }
    @Override
    public boolean isDone() {
        return subject.isDone();
    }
    @Override
    public HeadersFacade headers() {
        return requestWrapper.headers();
    }
    @Override
    public Any request() {
        return requestWrapper.body();
    }
    @Override
    public ResponseWrapperFacade toReply(Message message, ResponseStatusFacade status) {
        return requestWrapper.toReply(message, status);
    }
    @Override
    public ResponseWrapperFacade toExceptionalReply(Message message, Exception exception) {
        return requestWrapper.toExceptionalReply(message, exception);
    }
    @Override
    public Span span() {
        return span;
    }
    @Override
    public void addListener(Runnable listener, Executor executor) {
        subject.addListener(listener, executor);
    }
    @Override
    public void addCallback(FutureCallback<ApiResponseEntity<RESP>> callback, Executor executor) {
        subject.addCallback(new FutureCallback<ResponseWrapperFacade>() {
            @Override
            public void onSuccess(ResponseWrapperFacade result) {
                callback.onSuccess(new DefaultApiResponseEntity<>(result, type));
            }
            @Override
            public void onFailure(Throwable t) {
                callback.onFailure(t);
            }
        }, executor);
    }
    @Override
    public ApiResponse<RESP> thenVerifyOk(Executor executor) {
        return thenVerifyOkAndAccept(new Consumer<RESP>() {
            @Override
            public void accept(RESP t) {

            }
        }, executor);
    }
    @Override
    public ApiResponse<RESP> thenVerifyOkAndAccept(Consumer<RESP> callback, Executor executor) {
        SettableFuture<ResponseWrapperFacade> toReturn = SettableFuture.create();
        subject.addCallback(new FutureCallback<ResponseWrapperFacade>() {
            @Override
            public void onSuccess(ResponseWrapperFacade result) {
                try {
                    ReplyUtil.verifyOk(result);
                } catch (ApplicationException err) {
                    toReturn.setException(err);
                }

                try {
                    RESP body = result.unpack(type);
                    callback.accept(body);
                    toReturn.set(result);
                } catch (Exception err) {
                    toReturn.setException(err);
                }
            }
            @Override
            public void onFailure(Throwable t) {
                toReturn.setException(t);
            }
        }, executor);

        //
        // ~ new API response by contract
        //
        return new DefaultApiResponse<RESP>(toReturn, type, requestWrapper, span);
    }
}
