package com.turbospaces.executor;

import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor.AbortPolicy;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Supplier;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.PlatformExecutorService;
import com.turbospaces.common.PlatformUtil;

import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;

public class DefaultPlatformExecutorService
        extends AbstractBootstrapAware
        implements PlatformExecutorService, InitializingBean, DisposableBean, Supplier<ExecutorService> {
    private final String name;
    private final int minPoolSize;
    private final int maxPoolSize;
    private final RejectedExecutionHandler rejectPolicy;

    private ExecutorService executor;

    public DefaultPlatformExecutorService(String name, int minPoolSize, int maxPoolSize) {
        this.name = Objects.requireNonNull(name);
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
        this.rejectPolicy = new AbortPolicy();
    }
    public DefaultPlatformExecutorService(String name, int minPoolSize, int maxPoolSize, RejectedExecutionHandler rejectPolicy) {
        this.name = Objects.requireNonNull(name);
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
        this.rejectPolicy = Objects.requireNonNull(rejectPolicy);
    }
    @Override
    public void afterPropertiesSet() {
        ThreadFactoryBuilder factory = new ThreadFactoryBuilder();
        factory.setDaemon(false);
        factory.setNameFormat(name + "-%d");

        Duration ttl = bootstrap.props().APP_PLATFORM_MAX_IDLE.get();
        executor = new ThreadPoolExecutor(
                minPoolSize,
                maxPoolSize,
                ttl.toSeconds(),
                TimeUnit.SECONDS,
                new SynchronousQueue<Runnable>(),
                factory.build(),
                rejectPolicy);

        ExecutorServiceMetrics metrics = new ExecutorServiceMetrics(executor, name, Collections.emptyList());
        metrics.bindTo(bootstrap.meterRegistry());
    }
    @Override
    public void destroy() {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = bootstrap.props().APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();

        logger.debug("stopped executor-service: {} in {}", executor, stopWatch);
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    @VisibleForTesting
    public ExecutorService get() {
        return executor;
    }
}
