package com.turbospaces.dispatch;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;

import org.springframework.util.concurrent.ListenableFutureCallback;

import com.google.protobuf.Message;

import io.netty.util.AsciiString;
import io.opentracing.Span;

public interface QueuePostSpec<C> {
    String topic();
    Optional<AsciiString> routingKey();
    Optional<String> replyTo();
    UUID messageId();
    Message payload();
    Span parentSpan();
    Duration timeout();
    Optional<ListenableFutureCallback<C>> callback();

    public static abstract class AbstractBuilder<T, C> {
        protected String topic;
        protected AsciiString routingKey;
        protected String replyTo;
        protected Optional<UUID> messageId = Optional.empty();
        protected Duration timeout;
        protected T payload;
        protected Span parentSpan;
        protected ListenableFutureCallback<C> callback;

        protected AbstractBuilder(Duration timeout) {
            this.timeout = Objects.requireNonNull(timeout);
        }
        public AbstractBuilder<T, C> setTopic(String topic) {
            this.topic = Objects.requireNonNull(topic);
            return this;
        }
        public AbstractBuilder<T, C> setRoutingKey(AsciiString routingKey) {
            this.routingKey = Objects.requireNonNull(routingKey);
            return this;
        }
        public AbstractBuilder<T, C> setReplyTo(String replyTo) {
            this.replyTo = Objects.requireNonNull(replyTo);
            return this;
        }
        public AbstractBuilder<T, C> setMessageId(UUID messageId) {
            this.messageId = Optional.of(messageId);
            return this;
        }
        public AbstractBuilder<T, C> setPayload(T payload) {
            this.payload = Objects.requireNonNull(payload);
            return this;
        }
        public AbstractBuilder<T, C> setParentSpan(Span span) {
            this.parentSpan = Objects.requireNonNull(span);
            return this;
        }
        public AbstractBuilder<T, C> setCallback(ListenableFutureCallback<C> callback) {
            this.callback = Objects.requireNonNull(callback);
            return this;
        }
        public abstract QueuePostSpec<C> build();
    }
}
