package com.turbospaces.dispatch;

import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import org.apache.commons.lang3.exception.ExceptionUtils;

import com.google.common.util.concurrent.FutureCallback;
import com.google.protobuf.InvalidProtocolBufferException;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.rpc.ApiResponse;

import api.v1.ResponseWrapperFacade;
import io.opentracing.Scope;
import io.opentracing.Span;
import io.vavr.CheckedConsumer;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public abstract class AbstractSafeResponseConsumer implements Runnable, FutureCallback<ResponseWrapperFacade>, CheckedConsumer<ResponseWrapperFacade> {
    private final Bootstrap bootstrap;
    private final ApiResponse<?> apiResponse;
    private final Span span;

    public AbstractSafeResponseConsumer(ApiResponse<?> post, Bootstrap bootstrap) {
        this.apiResponse = Objects.requireNonNull(post);
        this.span = Objects.requireNonNull(post.span());
        this.bootstrap = Objects.requireNonNull(bootstrap);
    }
    @Override
    public final void run() {
        try (Scope activate = bootstrap.tracer().activateSpan(span)) {
            ResponseWrapperFacade respw = null;
            try {
                var input = apiResponse.get();
                respw = apiResponse.toReply(input.body(), input.status());
            } catch (ExecutionException | InvalidProtocolBufferException err) {
                Throwable cause = err.getCause();
                if (cause instanceof TimeoutException) {
                    log.warn(cause.getMessage(), cause); // ~ just warning

                    //
                    // ~ accept wrapped timeout exception properly
                    //
                    try {
                        onSuccess(apiResponse.toTimeoutReply((TimeoutException) cause));
                    } catch (Exception nested) {
                        log.error(nested.getMessage(), nested); // ~ raise sentry alert at least
                    }
                } else {
                    onFailure(cause);
                }
            } catch (InterruptedException err) {
                onFailure(err);
            }

            if (Objects.nonNull(respw)) {
                onSuccess(respw);
            }
        }
    }
    @Override
    public final void onSuccess(ResponseWrapperFacade result) {
        try {
            accept(result);
        } catch (Throwable t) {
            onFailure(t);
        }
    }
    @Override
    public void onFailure(Throwable t) {
        Throwable rootCause = ExceptionUtils.getRootCause(t);
        if (Objects.isNull(rootCause)) {
            rootCause = t;
        }
        log.error(rootCause.getMessage(), rootCause); // ~ raise sentry alert at least
    }
}
