package com.turbospaces.handlers;

import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Supplier;

import org.springframework.util.concurrent.ListenableFutureCallback;

import com.google.protobuf.Message;

import io.netty.util.AsciiString;
import io.opentracing.Span;

public interface QueuePostSpec<C> {
    String topic();
    int timeout(Supplier<Integer> fallback);
    Optional<AsciiString> routingKey();
    Optional<String> replyTo();
    UUID messageId();
    Message payload();
    Span parentSpan();
    Optional<ListenableFutureCallback<C>> callback();

    public static abstract class AbstractBuilder<T, C> {
        protected String topic;
        protected AsciiString routingKey;
        protected String replyTo;
        protected Optional<UUID> messageId = Optional.empty();
        protected Optional<Integer> timeout = Optional.empty();
        protected T payload;
        protected Span span;
        protected ListenableFutureCallback<C> callback;

        public AbstractBuilder<T, C> setTopic(String topic) {
            this.topic = Objects.requireNonNull(topic);
            return this;
        }
        public AbstractBuilder<T, C> setRoutingKey(AsciiString routingKey) {
            this.routingKey = Objects.requireNonNull(routingKey);
            return this;
        }
        public AbstractBuilder<T, C> setReplyTo(String replyTo) {
            this.replyTo = Objects.requireNonNull(replyTo);
            return this;
        }
        public AbstractBuilder<T, C> setTimeout(int timeout) {
            this.timeout = Optional.of(timeout);
            return this;
        }
        public AbstractBuilder<T, C> setMessageId(UUID messageId) {
            this.messageId = Optional.of(messageId);
            return this;
        }
        public AbstractBuilder<T, C> setPayload(T payload) {
            this.payload = Objects.requireNonNull(payload);
            return this;
        }
        public AbstractBuilder<T, C> setSpan(Span span) {
            this.span = Objects.requireNonNull(span);
            return this;
        }
        public AbstractBuilder<T, C> setCallback(ListenableFutureCallback<C> callback) {
            this.callback = Objects.requireNonNull(callback);
            return this;
        }
        public abstract QueuePostSpec<C> build();
    }
}
