package com.turbospaces.boot;

import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;

import org.slf4j.MDC;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.turbospaces.common.MDCUtil;
import com.turbospaces.executor.DirectPlatformExecutorService;

import io.vavr.CheckedFunction0;
import io.vavr.CheckedRunnable;

public interface PlatformExecutorService extends Executor {
    PlatformExecutorService DIRECT_EXECUTOR = new DirectPlatformExecutorService();

    default ListenableFuture<?> submit(CheckedRunnable action) {
        SettableFuture<Object> settable = SettableFuture.create();

        try {
            execute(new Runnable() {
                @Override
                public void run() {
                    try {
                        action.run();
                        settable.set(new Object());
                    } catch (Throwable err) {
                        settable.setException(err);
                    }
                }
            });
        } catch (RejectedExecutionException err) {
            settable.setException(err);
        }

        return settable;
    }
    default <T> ListenableFuture<T> submit(CheckedFunction0<T> action) {
        SettableFuture<T> settable = SettableFuture.create();

        try {
            execute(new Runnable() {
                @Override
                public void run() {
                    try {
                        settable.set(action.apply());
                    } catch (Throwable err) {
                        settable.setException(err);
                    }
                }
            });
        } catch (RejectedExecutionException err) {
            settable.setException(err);
        }

        return settable;
    }
    default Runnable wrapRunnable(Runnable task, Map<String, String> mdcContextMap) {
        return new Runnable() {
            @Override
            public void run() {
                MDCUtil.propagete(mdcContextMap);
                try {
                    task.run();
                } finally {
                    MDC.clear();
                }
            }
        };
    }
    default <V> Callable<V> wrapCallable(Callable<V> task, Map<String, String> mdcContextMap) {
        return new Callable<V>() {
            @Override
            public V call() throws Exception {
                MDCUtil.propagete(mdcContextMap);
                try {
                    return task.call();
                } finally {
                    MDC.clear();
                }
            }
        };
    }
}
