package com.turbospaces.healthchecks;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.service.common.RelationalServiceInfo;

import com.codahale.metrics.health.HealthCheck;
import com.turbospaces.cfg.ApplicationProperties;

public class RelationalHealthCheck extends HealthCheck {
    private final Logger logger = LoggerFactory.getLogger( getClass() );

    private final RelationalServiceInfo serviceInfo;
    private final Properties properties;

    public RelationalHealthCheck(ApplicationProperties props, RelationalServiceInfo serviceInfo) {
        this.serviceInfo = Objects.requireNonNull( serviceInfo );

        int timeout = (int) TimeUnit.SECONDS.toMillis( props.JDBC_CONNECTION_TIMEOUT.get() );

        properties = new Properties();
        properties.put( "connectTimeout", timeout );
        if ( serviceInfo.getUserName() != null ) {
            properties.put( "user", serviceInfo.getUserName() );
        }
        if ( serviceInfo.getPassword() != null ) {
            properties.put( "password", serviceInfo.getPassword() );
        }
    }
    @Override
    protected Result check() throws Exception {
        try {
            logger.debug( "checking JDBC connectivity on {}:{} ...", serviceInfo.getHost(), serviceInfo.getPort() );


            String jdbcUrl = serviceInfo.getJdbcUrl();
            Driver driver = DriverManager.getDriver( jdbcUrl );
            try (Connection connection = driver.connect( jdbcUrl, properties )) {
                String productVersion = connection.getMetaData().getDatabaseProductVersion();
                return HealthCheck.Result.healthy( "db version: " + productVersion );
            }
        }
        catch ( Exception err ) {
            logger.warn( err.getMessage(), err );
            return HealthCheck.Result.unhealthy( err );
        }
    }
}
