package com.turbospaces.healthchecks;

import java.net.InetSocketAddress;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.Cloud;
import org.springframework.cloud.service.common.CassandraServiceInfo;

import com.codahale.metrics.health.HealthCheck;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Session;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.boot.DisposableHealtchCheck;
import com.turbospaces.cassandra.CassandraSession;
import com.turbospaces.ups.UPSs;

public class CassandraHealthCheck extends DisposableHealtchCheck {
    private final Logger logger = LoggerFactory.getLogger( CassandraSession.class );
    private final Cluster cluster;

    public CassandraHealthCheck(Bootstrap boot) {
        this( UPSs.findRequiredServiceInfoByType( boot, CassandraServiceInfo.class ) );
    }
    public CassandraHealthCheck(Cloud cloud) {
        this( UPSs.findRequiredServiceInfoByType( cloud, CassandraServiceInfo.class ) );
    }
    public CassandraHealthCheck(CassandraServiceInfo info) {
        Cluster.Builder builder = Cluster.builder();
        info.getContactPoints().forEach( new Consumer<String>() {
            @Override
            public void accept(String contactPoint) {
                InetSocketAddress addr = new InetSocketAddress( contactPoint, info.getPort() );
                builder.addContactPointsWithPorts( addr );
            }
        } );
        this.cluster = builder.build();
    }
    @Override
    protected Result check() throws Exception {
        try {
            try (Session session = cluster.connect()) {
                return HealthCheck.Result.healthy( "connected hosts: " + session.getState().getConnectedHosts() );
            }
        }
        catch ( Exception err ) {
            logger.warn( err.getMessage(), err );
            return HealthCheck.Result.unhealthy( err );
        }
    }
    @Override
    public void preDestroy() throws Exception {
        cluster.close();
    }
}
