package com.turbospaces.cassandra;

import java.net.InetSocketAddress;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import org.apache.commons.lang3.StringUtils;
import org.springframework.cloud.Cloud;
import org.springframework.cloud.service.common.CassandraServiceInfo;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.QueryOptions;
import com.datastax.driver.core.SocketOptions;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ups.UPSs;

public abstract class CassandraUtil {
    public static Cluster cluster(ApplicationProperties props, Cloud cloud) {
        CassandraServiceInfo serviceInfo = UPSs.findRequiredServiceInfoByType( cloud, CassandraServiceInfo.class );
        return cluster( props, serviceInfo );
    }
    public static Cluster cluster(ApplicationProperties props, CassandraServiceInfo serviceInfo) {
        Cluster.Builder builder = Cluster.builder();

        // contact points
        serviceInfo.getContactPoints().forEach( new Consumer<String>() {
            @Override
            public void accept(String contactPoint) {
                builder.addContactPointsWithPorts( new InetSocketAddress( contactPoint, serviceInfo.getPort() ) );
            }
        } );
        if ( StringUtils.isNotEmpty( serviceInfo.getUsername() ) && StringUtils.isNotEmpty( serviceInfo.getPassword() ) ) {
            builder.withCredentials( serviceInfo.getUsername(), serviceInfo.getPassword() );
        }

        // socket options
        SocketOptions sopts = new SocketOptions().setKeepAlive( props.TCP_KEEP_ALIVE.get() )
                                                 .setReuseAddress( props.TCP_REUSE_ADDRESS.get() )
                                                 .setTcpNoDelay( props.TCP_NODELAY.get() )
                                                 .setConnectTimeoutMillis( (int) TimeUnit.SECONDS.toMillis( props.TCP_CONNECTION_TIMEOUT.get() ) );
        builder.withSocketOptions( sopts );

        // query options
        QueryOptions qopts = new QueryOptions();
        qopts.setConsistencyLevel( ConsistencyLevel.LOCAL_QUORUM );
        builder.withQueryOptions( qopts );

        return builder.build();
    }
    private CassandraUtil() {}
}
