package com.turbospaces.cassandra;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;

import com.datastax.driver.core.CloseFuture;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.RegularStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.ResultSetFuture;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;

public class DefaultCassandraSession implements CassandraSession {
    private final Map<String, PreparedStatement> stmts = new ConcurrentHashMap<>();
    private final Session target;

    public DefaultCassandraSession(Session target) {
        this.target = Objects.requireNonNull( target );
    }
    @Override
    public String getLoggedKeyspace() {
        return target.getLoggedKeyspace();
    }
    @Override
    public Session init() {
        return target.init();
    }
    @Override
    public ListenableFuture<Session> initAsync() {
        return target.initAsync();
    }
    @Override
    public ResultSet execute(String query) {
        return target.execute( query );
    }
    @Override
    public ResultSet execute(String query, Object... values) {
        return target.execute( query, values );
    }
    @Override
    public ResultSet execute(String query, Map<String, Object> values) {
        return target.execute( query, values );
    }
    @Override
    public ResultSet execute(Statement statement) {
        return target.execute( statement );
    }
    @Override
    public ResultSetFuture executeAsync(String query) {
        return target.executeAsync( query );
    }
    @Override
    public ResultSetFuture executeAsync(String query, Object... values) {
        return target.executeAsync( query, values );
    }
    @Override
    public ResultSetFuture executeAsync(String query, Map<String, Object> values) {
        return target.executeAsync( query, values );
    }
    @Override
    public ResultSetFuture executeAsync(Statement statement) {
        return target.executeAsync( statement );
    }
    @Override
    public PreparedStatement prepare(String query) {
        PreparedStatement stmt = stmts.get( query );
        if ( stmt == null ) {
            stmt = target.prepare( query );
            PreparedStatement prev = stmts.putIfAbsent( query, stmt );
            if ( prev != null ) {
                stmt = prev;
            }
        }
        return stmt;
    }
    @Override
    public PreparedStatement prepare(RegularStatement statement) {
        return target.prepare( statement );
    }
    @Override
    public ListenableFuture<PreparedStatement> prepareAsync(String query) {
        PreparedStatement stmt = stmts.get( query );
        if ( stmt == null ) {
            ListenableFuture<PreparedStatement> f = target.prepareAsync( query );
            f.addListener( new Runnable() {
                @Override
                public void run() {
                    try {
                        stmts.putIfAbsent( query, f.get() );
                    }
                    catch ( InterruptedException | ExecutionException err ) {
                        throw new RuntimeException( err );
                    }
                }
            }, MoreExecutors.directExecutor() );
            return f;
        }
        return Futures.immediateFuture( stmt );
    }
    @Override
    public ListenableFuture<PreparedStatement> prepareAsync(RegularStatement statement) {
        return target.prepareAsync( statement );
    }
    @Override
    public CloseFuture closeAsync() {
        return target.closeAsync();
    }
    @Override
    public void close() {
        target.close();
    }
    @Override
    public boolean isClosed() {
        return target.isClosed();
    }
    @Override
    public Cluster getCluster() {
        return target.getCluster();
    }
    @Override
    public State getState() {
        return target.getState();
    }
}
